#Updates W matrix for episodically consumed foods.
#Appendix A.9 and A.10 in Zhang, et al. (2011).
update_w_matrix <- function(w.matrix,
                            xbeta.u,
                            sigma.e,
                            recall.availability,
                            episodic.indicator.matrices,
                            num.subjects,
                            num.episodic,
                            num.daily,
                            num.recalls) {

  num.variables <- 2*num.episodic + num.daily

  inv.sigma.e <- solve(sigma.e)

  w.matrix.updated <- w.matrix
  for(day.k in seq_len(num.recalls)) {

    for(var.j in seq_len(2*num.episodic)) {

      food.j <- (var.j + 1) %/% 2
      observed <- (episodic.indicator.matrices[[day.k]][,food.j] == 1)

      #C1 and C2 in W matrix complete conditionals from Appendix A.9 and A.10 of Zhang, et al. (2011)
      c1 <- numeric(num.subjects)
      for(var.jj in seq_len(num.variables)) {

        if(var.jj == var.j) {

          c1 <- c1 + xbeta.u[[day.k]][,var.jj]*inv.sigma.e[var.jj, var.jj]
        } else {

          c1 <- c1 - (w.matrix.updated[[day.k]][,var.jj] - xbeta.u[[day.k]][,var.jj])*inv.sigma.e[var.j, var.jj]
        }
      }

      c2 <- 1/inv.sigma.e[var.j, var.j]

      mu <- c2*c1
      sigma <- sqrt(c2)

      if(var.j %% 2 == 1) {

        #Update W matrix column for indicator using the complete conditional from Appendix A.9 of Zhang, et al. (2011)
        truncation.values <- ifelse(observed, -mu/sigma, mu/sigma)
        trunc.normal <- truncated_normal(truncation.values)

        w.matrix.updated[[day.k]][observed, var.j] <- (mu[observed] + sigma*trunc.normal[observed])*recall.availability[observed, day.k]
        w.matrix.updated[[day.k]][!observed, var.j] <- (mu[!observed] - sigma*trunc.normal[!observed])*recall.availability[!observed, day.k]
      } else {

        #Update the W matrix column for amount using the complete conditional from Appendix A.10 of Zhang, et al. (2011)
        normals <- rnorm(num.subjects, mean=mu, sd=sigma)

        w.matrix.updated[[day.k]][!observed, var.j] <- normals[!observed]*recall.availability[!observed, day.k]
      }
    }
  }

  return(w.matrix.updated)
}


#Computes weighted W-XBeta-U multiplied by unweighted W-XBeta-U for all variable combinations, summed over all recalls.
#W-XBeta-U corresponds to the error (epsilon) term in Equation 3.5 of Zhang, et al. (2011).
#Pre-calculated component in complete conditionals for r, theta, and V elements.
#See Appendix A.5 of Zhang, et al. (2011).
#This is calculated before updating r, theta, and V in order to save time - sigma-e varies with each element, but the error terms change just once per iteration.
calculate_w_cross_residual_sum <- function(w.matrix,
                                           xbeta.u,
                                           recall.availability,
                                           subject.weighting,
                                           num.episodic,
                                           num.daily,
                                           num.recalls) {

  num.variables <- 2*num.episodic + num.daily
  w.cross.residual.sum <- matrix(0, nrow=num.variables, ncol=num.variables)
  for(day.k in seq_len(num.recalls)) {

    #calculate residual error terms W - XBeta - U
    w.minus.xbeta.u <- (w.matrix[[day.k]] - xbeta.u[[day.k]])*recall.availability[,day.k]

    #sum the cross products of the residuals
    w.cross.residual.sum <- w.cross.residual.sum + t(w.minus.xbeta.u*subject.weighting) %*% w.minus.xbeta.u
  }

  return(w.cross.residual.sum)
}



#Updates r matrix using Metropolis-Hastings step.
#See Appendix A.5 of Zhang, et al. (2011).
update_r_matrix <- function(r.matrix,
                            theta.matrix,
                            v.matrix,
                            w.cross.residual.sum,
                            recall.availability,
                            subject.weighting) {

  #if r matrix is null, then return it immediately (this is because there aren't two or more episodic foods)
  if(is.null(r.matrix)) return(r.matrix)

  #uniform spacing between possible r values
  r.spacing <- 2*(0.99)/200

  #discrete list of possible r-values (see Appendix A.5 of Zhang, et al. (2011))
  r.possible <- seq(-0.99, 0.99, r.spacing)

  #initialize current r matrix
  r.current <- r.matrix

  #loop through current r values to update them
  for(i in seq_along(r.current)) {

    #calculate minimum and maximum r
    r.min <- min(r.possible, r.current[i])
    r.max <- max(r.possible, r.current[i])

    #choose candidate r value:
    #1. If r[i] <= r.min, choose candidate from current r[i] and r[i] + 1 or 2 r spacings
    #2. If r[i] is between r.min and r.max, choose candidate from current r[i] and r[i] +/- 1 r spacing
    #3. If r[i] >= r.max, choose candidate from current r[i] and r[i] - 1 or 2 r spacings

    #random uniform number to randomly select candidate from options
    candidate.choice <- runif(1)

    if(r.current[i] <= r.min) {
      candidate <- (r.current[i])*(candidate.choice <= 1/3) +
                        (r.current[i] + r.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                        (r.current[i] + 2*r.spacing)*(candidate.choice > 2/3)
    } else if(r.current[i] >= r.max) {
      candidate <- (r.current[i])*(candidate.choice <= 1/3) +
                        (r.current[i] - r.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                        (r.current[i] - 2*r.spacing)*(candidate.choice > 2/3)
    } else {
      candidate <- (r.current[i])*(candidate.choice <= 1/3) +
                        (r.current[i] + r.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                        (r.current[i] - r.spacing)*(candidate.choice > 2/3)
    }

    #if r candidate is between -1 and 1, perform Metropolis-Hastings step to determine whether to accept it
    #otherwise, automatically reject the candidate and the current r is kept the same
    if(candidate >= -1 && candidate <= 1) {

      #create candidate r matrix
      r.candidate <- r.current
      r.candidate[i] <- candidate

      #V matrix based on current and candidate r values
      v.current <- update_v_indicator(v.matrix, r.current, theta.matrix)
      v.candidate <- update_v_indicator(v.matrix, r.candidate, theta.matrix)

      #sigma-e based on current and candidate r values
      sigma.e.current <- v.current %*% t(v.current)
      sigma.e.candidate <- v.candidate %*% t(v.candidate)

      #inverse sigma.e for current and candidate values
      inv.sigma.e.current <- solve(sigma.e.current)
      inv.sigma.e.candidate <- solve(sigma.e.candidate)

      #weighted sum of number of recalls across all subjects
      num.recalls.per.subject <- rowSums(recall.availability)
      weighted.sum.recalls <- sum(num.recalls.per.subject * subject.weighting)

      # #current and candidate log-likelihood for Metropolis-Hastings step (Appendix A.5 of Zhang, et al. (2011))
      current.log.likelihood <- -0.5*weighted.sum.recalls*log(1 - r.current[i]^2) - 0.5*sum(inv.sigma.e.current * w.cross.residual.sum)
      candidate.log.likelihood <- -0.5*weighted.sum.recalls*log(1 - r.candidate[i]^2) - 0.5*sum(inv.sigma.e.candidate * w.cross.residual.sum)

      #calculate Metropolis-Hastings acceptance probability
      acceptance.probability <- min(1, exp(candidate.log.likelihood - current.log.likelihood))

      #current r value is replaced with the candidate with probability equal to the acceptance probability, otherwise the current r value is kept
      acceptance.choice <- runif(1)
      r.current[i] <- r.candidate[i]*(acceptance.choice <= acceptance.probability) + r.current[i]*(acceptance.choice > acceptance.probability)
    }
  }

  return(r.current)
}



#Updates theta matrix using Metropolis-Hastings step.
#See Appendix A.5 of Zhang, et al. (2011).
update_theta_matrix <- function(theta.matrix,
                                r.matrix,
                                v.matrix,
                                w.cross.residual.sum) {

  #if theta matrix is null, then return it immediately (this is because there aren't two or more episodic foods)
  if(is.null(theta.matrix)) return(theta.matrix)

  #uniform spacing between possible theta values
  pi <- acos(-1)
  theta.spacing <- 2*pi*(0.99)/200

  #discrete list of possible theta values (see Appendix A.5 of Zhang, et al. (2011))
  theta.possible <- seq(-pi, pi, theta.spacing)

  #initialize current theta matrix
  theta.current <- theta.matrix

  #loop through current theta values to update them
  for(i in seq_along(theta.current)) {

    #calculate minimum and maximum theta
    theta.min <- min(theta.possible, theta.current[i])
    theta.max <- max(theta.possible, theta.current[i])

    #choose candidate theta value:
    #1. If theta[i] <= theta.min, choose candidate from current theta[i] and theta[i] + 1 or 2 theta spacings
    #2. If theta[i] is between theta.min and theta.max, choose candidate from current theta[i] and theta[i] +/- 1 theta spacing
    #3. If theta[i] >= theta.max, choose candidate from current theta[i] and theta[i] - 1 or 2 theta spacings

    #random uniform number to randomly select candidate from options
    candidate.choice <- runif(1)

    if(theta.current[i] <= theta.min) {
      candidate <- (theta.current[i])*(candidate.choice <= 1/3) +
                            (theta.current[i] + theta.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                            (theta.current[i] + 2*theta.spacing)*(candidate.choice > 2/3)
    } else if(theta.current[i] >= theta.max) {
      candidate <- (theta.current[i])*(candidate.choice <= 1/3) +
                            (theta.current[i] - theta.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                            (theta.current[i] - 2*theta.spacing)*(candidate.choice > 2/3)
    } else {
      candidate <- (theta.current[i])*(candidate.choice <= 1/3) +
                            (theta.current[i] + theta.spacing)*(candidate.choice > 1/3 && candidate.choice <= 2/3) +
                            (theta.current[i] - theta.spacing)*(candidate.choice > 2/3)
    }

    #create candidate theta matrix
    theta.candidate <- theta.current
    theta.candidate[i] <- candidate

    #V matrix based on current and candidate theta values
    v.current <- update_v_indicator(v.matrix, r.matrix, theta.current)
    v.candidate <- update_v_indicator(v.matrix, r.matrix, theta.candidate)

    #sigma-e based on current and candidate theta values
    sigma.e.current <- v.current %*% t(v.current)
    sigma.e.candidate <- v.candidate %*% t(v.candidate)

    #inverse sigma.e for current and candidate values
    inv.sigma.e.current <- solve(sigma.e.current)
    inv.sigma.e.candidate <- solve(sigma.e.candidate)

    #current and candidate log-likelihood for Metropolis-Hastings step (Appendix A.5 of Zhang, et al. (2011))
    current.log.likelihood <- -0.5*sum(inv.sigma.e.current * w.cross.residual.sum)
    candidate.log.likelihood <- -0.5*sum(inv.sigma.e.candidate * w.cross.residual.sum)

    #calculate Metropolis-Hastings acceptance probability
    acceptance.probability <- min(1, exp(candidate.log.likelihood - current.log.likelihood))

    #current theta value is replaced with the candidate with probability equal to the acceptance probability, otherwise the current theta value is kept
    acceptance.choice <- runif(1)
    theta.current[i] <- theta.candidate[i]*(acceptance.choice <= acceptance.probability) + theta.current[i]*(acceptance.choice > acceptance.probability)
  }

  return(theta.current)
}



#Updates V matrix in three parts:
#1. Diagonal elements for amount variables
#2. Off-diagonal elements for amount variables
#3. Elements for indicator variables
update_v_matrix <- function(v.matrix,
                            r.matrix,
                            theta.matrix,
                            w.cross.residual.sum,
                            recall.availability,
                            subject.weighting,
                            num.episodic,
                            num.daily) {

  v.updated <- v.matrix

  #1. Diagonal elements for amount variables
  v.updated <- update_v_amount_diagonal(v.matrix=v.updated,
                                        r.matrix=r.matrix,
                                        theta.matrix=theta.matrix,
                                        w.cross.residual.sum=w.cross.residual.sum,
                                        recall.availability=recall.availability,
                                        subject.weighting=subject.weighting,
                                        num.episodic=num.episodic,
                                        num.daily=num.daily)

  #2. Off-diagonal elements for amount variables
  v.updated <- update_v_amount_off_diagonal(v.matrix=v.updated,
                                            r.matrix=r.matrix,
                                            theta.matrix=theta.matrix,
                                            w.cross.residual.sum=w.cross.residual.sum,
                                            num.episodic=num.episodic,
                                            num.daily=num.daily)

  #3. Elements for indicator variables
  v.updated <- update_v_indicator(v.matrix=v.updated,
                                  r.matrix=r.matrix,
                                  theta.matrix=theta.matrix)

  return(v.updated)
}


#Updates V matrix diagonal elements for amount variables using Metropolis-Hastings step.
#See Vqq complete conditional from Zhang, et al. (2011).
update_v_amount_diagonal <- function(v.matrix,
                                     r.matrix,
                                     theta.matrix,
                                     w.cross.residual.sum,
                                     recall.availability,
                                     subject.weighting,
                                     num.episodic,
                                     num.daily) {

  #initialize current V matrix
  v.current <- v.matrix

  #loop through amount diagonals and update them
  episodic.amounts <- seq(from=2, by=2, length=num.episodic)
  daily.amounts <- seq(from=2*num.episodic+1, length=num.daily)
  for(i in c(episodic.amounts, daily.amounts)) {

    #extract candidate value for V[i,i]
    current <- v.current[i,i]
    candidate <- runif(1, min=current-0.2, max=current+0.2)

    #if candidate is between -3 and 3, perform Metropolis-Hastings step to determine whether to accept it
    #otherwise, automatically reject the candidate and the current value is kept
    if(candidate >= -3 && candidate <= 3) {

      #replace V[i,i] in candidate V matrix with candidate value
      v.candidate <- v.current
      v.candidate[i,i] <- candidate

      #proposed v matrices based on current and candidate values
      v.proposed.current <- update_v_indicator(v.current, r.matrix, theta.matrix)
      v.proposed.candidate <- update_v_indicator(v.candidate, r.matrix, theta.matrix)

      #proposed sigma-e from current and candidate values
      sigma.e.current <- v.proposed.current %*% t(v.proposed.current)
      sigma.e.candidate <- v.proposed.candidate %*% t(v.proposed.candidate)

      #inverse of the proposed sigma-e
      inv.sigma.e.current <- solve(sigma.e.current)
      inv.sigma.e.candidate <- solve(sigma.e.candidate)

      #weighted sum of number of recalls across all subjects
      num.recalls.per.subject <- rowSums(recall.availability)
      weighted.sum.recalls <- sum(num.recalls.per.subject * subject.weighting)

      #current and candidate log-likelihood for Metropolis-Hastings step (Appendix A.5 of Zhang, et al. (2011))
      current.log.likelihood <- -0.5*weighted.sum.recalls*log(v.current[i,i]^2) - 0.5*sum(inv.sigma.e.current * w.cross.residual.sum)
      candidate.log.likelihood <- -0.5*weighted.sum.recalls*log(v.candidate[i,i]^2) - 0.5*sum(inv.sigma.e.candidate * w.cross.residual.sum)

      #calculate Metropolis-Hastings acceptance probability
      acceptance.probability <- min(1, exp(candidate.log.likelihood - current.log.likelihood))

      #current value is replaced with the candidate with probability equal to the acceptance probability, otherwise the current value is kept
      acceptance.choice <- runif(1)
      v.current[i,i] <- v.candidate[i,i]*(acceptance.choice <= acceptance.probability) + v.current[i,i]*(acceptance.choice > acceptance.probability)
    }
  }

  return(v.current)
}



#Updates V matrix off-diagonal elements for amount variables using Metropolis-Hastings step.
#see sample Vpq complete conditional from Zhang, et al. (2011).
update_v_amount_off_diagonal <- function(v.matrix,
                                         r.matrix,
                                         theta.matrix,
                                         w.cross.residual.sum,
                                         num.episodic,
                                         num.daily) {

  #initialize current V matrix
  v.current <- v.matrix

  #loop through every row of V matrix corresponding to amount variables and update off-diagonal elements
  episodic.amounts <- seq(from=2, by=2, length=num.episodic)
  daily.amounts <- seq(from=2*num.episodic+1, length=num.daily)
  for(i in c(episodic.amounts, daily.amounts)) {

    #loop through off-diagonal elements and update them
    if(i <= 2*num.episodic) {
      off.diagonal <- seq_len(i-2)
    } else {
      off.diagonal <- seq_len(i-1)
    }

    for(j in off.diagonal) {

      #Extract candidate value for V[i,j]
      current <- v.current[i,j]
      candidate <- runif(1, min=current-0.2, max=current+0.2)

      #if candidate is between -3 and 3, perform Metropolis-Hastings step to determine whether to accept it
      #otherwise, automatically reject the candidate and the current value is kept
      if(candidate >= -3 && candidate <= 3) {

        #replace V[i,j] in candidate V matrix with candidate value
        v.candidate <- v.current
        v.candidate[i,j] <- candidate

        #proposed v matrices based on current and candidate values
        v.proposed.current <- update_v_indicator(v.current, r.matrix, theta.matrix)
        v.proposed.candidate <- update_v_indicator(v.candidate, r.matrix, theta.matrix)

        #proposed sigma-e from current and candidate values
        sigma.e.current <- v.proposed.current %*% t(v.proposed.current)
        sigma.e.candidate <- v.proposed.candidate %*% t(v.proposed.candidate)

        #inverse of the proposed sigma-e
        inv.sigma.e.current <- solve(sigma.e.current)
        inv.sigma.e.candidate <- solve(sigma.e.candidate)

        #current and candidate log-likelihood for Metropolis-Hastings step (Appendix A.5 of Zhang, et al. (2011))
        current.log.likelihood <- -0.5*sum(inv.sigma.e.current * w.cross.residual.sum)
        candidate.log.likelihood <- -0.5*sum(inv.sigma.e.candidate * w.cross.residual.sum)

        #calculate Metropolis-Hastings acceptance probability
        acceptance.probability <- min(1, exp(candidate.log.likelihood - current.log.likelihood))

        #current value is replaced with the candidate with probability equal to the acceptance probability, otherwise the current value is kept
        acceptance.choice <- runif(1)
        v.current[i,j] <- v.candidate[i,j]*(acceptance.choice <= acceptance.probability) + v.current[i,j]*(acceptance.choice > acceptance.probability)
      }
    }
  }

  return(v.current)
}




#Updating V matrix indicator variables for both the main Gibbs sampler and for the current/candidate sigma-e in the Metropolis-Hastings steps for r, theta, and v elements.
#See section 3.2 of Zhang, et al. (2011) for the formula used in this function.
update_v_indicator <- function(v.matrix,
                               r.matrix,
                               theta.matrix) {

  #updates are only needed if there are 2 or more episodic variables
  if(is.null(r.matrix)) return(v.matrix)

  #initialize update V matrix to current V matrix
  v.updated <- v.matrix

  for(q in seq_along(r.matrix)) {

    for(p in seq_len(2*q - 1)) {

      if(p == 1) {
        v.updated[2*q+1,p] <- r.matrix[q]*sin(theta.matrix[p+(q-1)^2])
      } else {
        v.updated[2*q+1,p] <- r.matrix[q]*prod(cos(theta.matrix[1:(p-1) + (q-1)^2]))*sin(theta.matrix[p+(q-1)^2])
      }

      v.updated[2*q+1,2*q] <- r.matrix[q] * prod(cos(theta.matrix[1:p+(q-1)^2]))
    }

    v.updated[2*q+1, 2*q+1] <- sqrt(1 - (r.matrix[q])^2)
    v.updated[2*q+2, 2*q+1] <- -sum(v.updated[2*q+1,1:(2*q)] * v.updated[2*q+2,1:(2*q)])/v.updated[2*q+1,2*q+1]
  }

  return(v.updated)

}



#Updates Sigma-u matrix.
#Uses complete conditional in Appendix A.6 of Zhang, et al. (2011).
update_sigma_u <- function(sigma.u,
                           sigma.u.prior,
                           u.matrix,
                           subject.weighting,
                           sigma.u.constant,
                           num.subjects) {

  #check here for constant sigma-u parameter - return sigma-u unchanged if it is true
  if(sigma.u.constant) {
    return(sigma.u)
  }

  #sigma-u prior degrees of freedom
  sigma.u.prior.df <- nrow(sigma.u.prior) + 2

  #calculate inverse Wishart matrix parameter
  inv.wishart <- (sigma.u.prior.df - nrow(sigma.u) - 1)*sigma.u.prior + t(subject.weighting*u.matrix) %*% u.matrix

  #calculate inverse Wishart degrees of freedom
  wishart.df <- sigma.u.prior.df + num.subjects

  #Adjust inverse Wishart matrix diagonals and degrees of freedom by 0.0011 times the number of subjects
  inv.wishart <- inv.wishart + 0.0011*num.subjects*diag(ncol(sigma.u))
  wishart.df <- wishart.df + 0.0011*num.subjects

  #convert adjust inverse Wishart matrix into a symmetric matrix
  inv.wishart <- (inv.wishart + t(inv.wishart))/2

  #take the inverse of the inverse Wishart matrix to compute the matrix parameter for the forward Wishart function
  fwd.wishart <- solve(inv.wishart)

  #calculate inverse sigma-u using the forward Wishart function
  inv.sigma.u <- rWishart(1, df=wishart.df, Sigma=fwd.wishart)[,,1]
  dim(inv.sigma.u) <- dim(sigma.u)
  dimnames(inv.sigma.u) <- dimnames(sigma.u)

  #calculate sigma-u
  sigma.u.updated <- solve(inv.sigma.u)
  return(sigma.u.updated)
}


#Updates U matrix.
#Uses complete conditional in Appendix A.8 of Zhang, et al. (2011).
#Uses a Metropolis-Hastings step to sample values for never-consumers when they are present.
update_u_matrix <- function(sigma.u,
                            sigma.e,
                            w.matrix,
                            xbeta,
                            recall.availability,
                            num.subjects,
                            num.episodic,
                            num.daily,
                            num.recalls,
                            never.consumers.first.episodic,
                            u.matrix,
                            conni1) {

  #initialize candidate U matrix to input U matrix
  u.matrix.current <- u.matrix
  u.matrix.candidate <- u.matrix

  #calculating inverse of sigma-e and sigma-u
  inv.sigma.e <- solve(sigma.e)
  inv.sigma.u <- solve(sigma.u)

  #calculate total recalls per subject
  recalls.per.subject <- rowSums(recall.availability)

  num.variables <- 2*num.episodic + num.daily

  #Calculating C1 in complete conditional
  w.minus.xbeta.sum <- matrix(0, nrow=num.subjects, ncol=num.variables)
  for(day.k in seq_len(num.recalls)) {

    w.minus.xbeta.sum <- w.minus.xbeta.sum + (w.matrix[[day.k]] - xbeta[[day.k]])*recall.availability[,day.k]
  }
  c1 <- w.minus.xbeta.sum %*% inv.sigma.e

  #calculate candidate U matrix for each number of recalls
  for(count.k in seq_len(num.recalls)) {

    if(any(recalls.per.subject == count.k)) {

      #calculating C2 in complete conditional
      c2 <- solve(inv.sigma.u + count.k*inv.sigma.e)

      #calculate the mean of the candidate U matrix
      u.matrix.mean <- c1[recalls.per.subject == count.k,] %*% c2

      #calculate standard deviation of candidate U matrix by taking the matrix square root of C2
      c2.eigen <- eigen(c2)
      u.matrix.std.dev <- c2.eigen$vectors %*% diag(sqrt(c2.eigen$values),nrow=nrow(c2),ncol=ncol(c2)) %*% t(c2.eigen$vectors)

      #update the candidate U matrix for the current number of recalls according to the complete conditional in Appendix A.8 of Zhang, et al. (2011)
      #normal distribution with a mean of C1 * C2 and a standard deviation of sqrt(C2)
      normals <- matrix(rnorm(sum(recalls.per.subject == count.k)*num.variables), nrow=sum(recalls.per.subject == count.k), ncol=num.variables)
      u.matrix.candidate[recalls.per.subject == count.k,] <- u.matrix.mean + normals %*% u.matrix.std.dev
    }
  }


  #if never-consumers are allowed, perform Metropolis-Hastings step for the first episodic variable for each never-consumer and automatically accept consumers
  #otherwise, accept entire candidate U matrix as-is
  if(never.consumers.first.episodic && any(conni1 <= 0)) {

    #never-consumer subjects
    never.consumers <- (conni1 <= 0)
    num.never.consumers <- sum(never.consumers)

    #calculate log-likelihood for current and candidate U matrices
    log.likelihood.candidate <- numeric(num.subjects)
    log.likelihood.current <- numeric(num.subjects)

    for(day.k in seq_len(num.recalls)) {

      xbeta.u.candidate <- xbeta[[day.k]][never.consumers,1] + u.matrix.candidate[never.consumers,1]
      xbeta.u.current <- xbeta[[day.k]][never.consumers,1] + u.matrix.current[never.consumers,1]

      density.candidate <- pmax(1 - pnorm(xbeta.u.candidate), sqrt(.Machine$double.eps))
      density.current <- pmax(1 - pnorm(xbeta.u.current), sqrt(.Machine$double.eps))

      log.likelihood.candidate[never.consumers] <- log.likelihood.candidate[never.consumers] - recall.availability[never.consumers,day.k]*log(density.candidate)
      log.likelihood.current[never.consumers] <- log.likelihood.current[never.consumers] - recall.availability[never.consumers,day.k]*log(density.current)
    }

    #calculate acceptance probabilites for each subject
    #log-likelihood ratios above zero are set to zero (corresponding to an acceptance probability of 1)
    log.likelihood.ratio <- pmin(log.likelihood.candidate - log.likelihood.current, 0)
    acceptance.probability <- exp(log.likelihood.ratio)

    #accept the candidate for each subject with a probability equal to the acceptance probability
    #otherwise reject the candidate and use the original U matrix value
    acceptance.choice <- runif(num.subjects)
    u.matrix.updated <- u.matrix.candidate*(acceptance.choice <= acceptance.probability) + u.matrix.current*(acceptance.choice > acceptance.probability)

  } else {
    u.matrix.updated <- u.matrix.candidate
  }

  return(u.matrix.updated)
}



#Updates betas for each variable.
#Uses complete conditional defined in Appendix A.7 of Zhang, et al. (2011).
#If never-consumers are allowed, uses a Metropolis-Hastings step to sample beta values for the first episodic variable.
update_beta <- function(weighted.covariate.matrices,
                        weighted.covariate.squared.sums,
                        recall.availability,
                        w.matrix,
                        u.matrix,
                        sigma.e,
                        xbeta,
                        beta.mean.prior,
                        beta.covariance.prior,
                        num.subjects,
                        num.episodic,
                        num.daily,
                        num.recalls,
                        never.consumers.first.episodic,
                        conni1,
                        beta1,
                        covariate.matrices) {

  #calculate inverse sigma-e
  inv.sigma.e <- solve(sigma.e)

  #update betas for each episodic and daily variable
  num.variables <- 2*num.episodic + num.daily
  beta.updated <- vector(mode="list", length=num.variables)
  for(var.j in seq_len(num.variables)) {

    #number of covariates for the current variable
    num.covariates <- nrow(weighted.covariate.matrices[[var.j]][[1]])

    #calculate the C1 term in the beta complete conditional from Appendix A.7 of Zhang, et al. (2011)
    c1 <- solve(beta.covariance.prior[[var.j]], beta.mean.prior[[var.j]])
    for(var.jj in seq_len(num.variables)) {

      for(day.k in seq_len(num.recalls)) {

        if(var.jj == var.j) {

          c1 <- c1 + as.vector(weighted.covariate.matrices[[var.j]][[day.k]] %*% (w.matrix[[day.k]][,var.j] - u.matrix[,var.j]))*inv.sigma.e[var.j,var.j]
        } else {

          c1 <- c1 + as.vector(weighted.covariate.matrices[[var.j]][[day.k]] %*% (w.matrix[[day.k]][,var.jj] - xbeta[[day.k]][,var.jj] - u.matrix[,var.jj]))*inv.sigma.e[var.jj,var.j]
        }
      }
    }

    #calculate the C2 term in the beta complete conditional from Appendix A.7 of Zhang, et al. (2011)
    inverse.c2 <- solve(beta.covariance.prior[[var.j]]) + inv.sigma.e[var.j,var.j]*weighted.covariate.squared.sums[[var.j]]
    c2 <- solve(inverse.c2)


    #if never-consumers are allowed, perform Metropolis-Hastings step to update beta for first episodic variable
    #otherwise, use the complete conditional in Appendix A.7 of Zhang, et al. (2011)
    if(never.consumers.first.episodic && var.j == 1) {

      #never-consumer subjects
      never.consumers <- (conni1 <= 0)

      #current beta for first episodic variable
      beta1.current <- beta1

      #candidate beta for first episodic variable
      beta1.variance <- 2*c2
      beta1.variance.eigen <- eigen(beta1.variance)
      beta1.std.dev <- beta1.variance.eigen$vectors %*% diag(sqrt(beta1.variance.eigen$values),nrow=nrow(beta1.variance),ncol=ncol(beta1.variance)) %*% t(beta1.variance.eigen$vectors)

      beta1.candidate <- as.vector(beta1.current + beta1.std.dev %*% rnorm(num.covariates))

      #calculate log-likelihood for candidate and current beta
      log.likelihood.candidate <- as.vector(t(c1) %*% beta1.candidate - 0.5*t(beta1.candidate) %*% inverse.c2 %*% beta1.candidate)
      log.likelihood.current <- as.vector(t(c1) %*% beta1.current - 0.5*t(beta1.current) %*% inverse.c2 %*% beta1.current)

      for(day.k in seq_len(num.recalls)) {

        xbeta.u1.candidate <- as.vector(covariate.matrices[[1]][[day.k]][never.consumers,] %*% beta1.candidate + u.matrix[never.consumers,1])
        xbeta.u1.current <- as.vector(covariate.matrices[[1]][[day.k]][never.consumers,] %*% beta1.current + u.matrix[never.consumers,1])

        density.candidate <- pmax(1 - pnorm(xbeta.u1.candidate), sqrt(.Machine$double.eps))
        density.current <- pmax(1 - pnorm(xbeta.u1.current), sqrt(.Machine$double.eps))

        log.likelihood.candidate <- log.likelihood.candidate - sum(recall.availability[never.consumers,day.k]*log(density.candidate))
        log.likelihood.current <- log.likelihood.current - sum(recall.availability[never.consumers,day.k]*log(density.current))
      }

      #calculating acceptance probability
      acceptance.probability <- min(1, exp(log.likelihood.candidate - log.likelihood.current))

      #select candidate beta1 with probability equal to the acceptance probability, otherwise reject the candidate and keep the current beta1
      acceptance.choice <- runif(1)
      beta.updated[[1]] <- beta1.candidate*(acceptance.choice <= acceptance.probability) + beta1.current*(acceptance.choice > acceptance.probability)

    } else {

      #calculate the mean of the updated beta
      beta.mean <- c2 %*% c1

      #calculate the standard deviation of the updated beta by taking the matrix square root of C2
      c2.eigen <- eigen(c2)
      beta.std.dev <- c2.eigen$vectors %*% diag(sqrt(c2.eigen$values),nrow=nrow(c2),ncol=ncol(c2)) %*% t(c2.eigen$vectors)

      #calculate updated beta for this variable
      #normal distribution with a mean of C2*C1 and a standard deviation of sqrt(C2)
      beta.updated[[var.j]] <- as.vector(beta.mean + beta.std.dev %*% rnorm(num.covariates))
    }
    names(beta.updated[[var.j]]) <- names(beta.mean.prior[[var.j]])
  }
  names(beta.updated) <- names(beta.mean.prior)

  return(beta.updated)
}

#Updates conditional Ni for the first episodic food for each subject.
#Negative for never-consumers and positive for consumers.
#Uses complete conditional in Appendix A.2 of Bhadra, et al. (2020).
update_conni1 <- function(never.consumers.first.episodic,
                          alpha1,
                          consumer.probabilities,
                          xbeta.u,
                          never.consumer.covariate.matrix,
                          episodic.indicator.matrices,
                          recall.availability,
                          num.subjects,
                          num.recalls) {

  if(!never.consumers.first.episodic) {

    return(NULL)
  }

  #probability that each subject reports no consumption days
  probability.no.consumption.days <- rep(1, num.subjects)
  for(day.k in seq_len(num.recalls)) {

    probability.no.consumption.day.k <- (1 - pnorm(xbeta.u[[day.k]][,1]))^recall.availability[,day.k]
    probability.no.consumption.days <- probability.no.consumption.days*probability.no.consumption.day.k
  }

  #probability that each subject is a never-consumer given that they report no consumption
  never.consumer.probabilities <- 1 - consumer.probabilities
  conditional.never.consumer.probabilities <- never.consumer.probabilities/(never.consumer.probabilities + consumer.probabilities*probability.no.consumption.days)

  #estimate consumer probabilities
  g.alpha <- as.vector(never.consumer.covariate.matrix %*% alpha1)

  #find subjects with with with no reported consumption
  no.consumption.indicator <- rep(1, num.subjects)
  for(day.k in seq_len(num.recalls)) {

    no.consumption.indicator.day.k <- 1 - episodic.indicator.matrices[[day.k]][,1]
    no.consumption.indicator <- no.consumption.indicator*no.consumption.indicator.day.k
  }
  any.consumption.indicator <- 1 - no.consumption.indicator

  #choose never-consumers based on conditional never-consumer probabilities
  never.consumer.choice <- runif(num.subjects)
  never.consumers <- no.consumption.indicator & (never.consumer.choice <= conditional.never.consumer.probabilities)

  #truncated normals
  truncation.values <- ifelse(never.consumers, g.alpha, -g.alpha)
  trunc.normal <- truncated_normal(truncation.values)

  #update conni1
  conni1.updated <- numeric(num.subjects)
  conni1.updated[!never.consumers] <- g.alpha[!never.consumers] + trunc.normal[!never.consumers]
  conni1.updated[never.consumers] <- g.alpha[never.consumers] - trunc.normal[never.consumers]

  return(conni1.updated)
}

#Updates alpha1.
#Uses complete conditional in Appendix A.3 of Bhadra, et al. (2020).
update_alpha1 <- function(never.consumers.first.episodic,
                          alpha1.mean.prior,
                          alpha1.covariance.prior,
                          conni1,
                          never.consumer.covariate.matrix,
                          subject.weighting) {

  if(!never.consumers.first.episodic) {

    return(NULL)
  }

  #Calculate C1 (mean) in the complete conditional for alpha in Appendix A.3 of Bhadra, et al. (2020)
  weighted.never.consumer.covariates <- t(subject.weighting*never.consumer.covariate.matrix)
  c1.first.term <- solve(alpha1.covariance.prior) %*% alpha1.mean.prior
  c1.second.term <- weighted.never.consumer.covariates %*% conni1
  c1 <- c1.first.term + c1.second.term

  #Calculate C2 (variance) in the complete conditional for alpha in Appendix A.3 of Bhadra, et al. (2020)
  weighted.sum.covariates.squared <- weighted.never.consumer.covariates %*% never.consumer.covariate.matrix
  inverse.c2 <- solve(alpha1.covariance.prior) + weighted.sum.covariates.squared
  c2 <- solve(inverse.c2)

  #calculate the standard deviation of alpha (square root of C2)
  c2.eigen <- eigen(c2)
  alpha1.std.dev <- c2.eigen$vectors %*% diag(sqrt(c2.eigen$values),nrow=nrow(c2),ncol=ncol(c2)) %*% t(c2.eigen$vectors)

  #calculate updated alpha1
  alpha1.mean <- c2 %*% c1
  alpha1.updated <- as.vector(alpha1.mean + alpha1.std.dev %*% rnorm(length(alpha1.mean.prior)))
  names(alpha1.updated) <- names(alpha1.mean.prior)

  return(alpha1.updated)
}

#Updates consumer probabilities.
#From page 7 of Bhadra, et al. (2020).
update_consumer_probabilities <- function(never.consumers.first.episodic,
                                          never.consumer.covariate.matrix,
                                          alpha1) {

  if(!never.consumers.first.episodic) {

    return(NULL)
  }

  g.alpha <- as.vector(never.consumer.covariate.matrix %*% alpha1)
  consumer.probabilities <- pnorm(g.alpha)

  return(consumer.probabilities)
}
