#' Simulate Usual Intakes
#'
#' @description Simulates usual intakes for subjects that represent the
#'   distribution of true usual intakes.
#'
#' @details Usual intakes are simulated for each subject in the
#'   `distrib.population` dataset for different values of the random effects
#'   ('u'). The `num.simulated.u` parameter determines how many values of 'u'
#'   are simulated per subject. The simulated 'u' values are drawn from a
#'   multivariate normal distribution with a mean of zero and
#'   variance/covariance matrix equal to the posterior mean of `sigma.u` in
#'   `multivar.mcmc.model`.
#'
#'   Optionally, saved values of 'u' drawn after the MCMC chain conditional on
#'   the posterior means of the parameters can be used instead of simulating new
#'   values of 'u'. This must be done when using regression calibration for
#'   measurement error correction (see "Regression Calibration").
#'
#'   Simulated usual intakes are output for each variable in the MCMC model. The
#'   usual intake for daily variables is simply the long-term average amount
#'   consumed. Usual intakes for episodic variables are the product of the
#'   long-term probability of consumption and the long-term amount consumed
#'   given consumption. For episodic variables, the usual intake and the
#'   separate probability and amount components are included in the output.
#'
#' @section Regression Calibration: The simulated usual intakes from
#'   `nci_multivar_distrib()` can be used in regression calibration for
#'   measurement error correction. To do this, saved random effect ('u')
#'   matrices conditional on the posterior means of the MCMC parameters must be
#'   used in place of simulated values of 'u'. The values of `id` must match
#'   with `mcmc.subjects` in `multivar.mcmc.model` to apply the saved post-MCMC
#'   'u' values. New values of 'u' are simulated for unmatched observations. Use
#'   \code{vignette("regression_calibration", package="ncimultivar")} for a full
#'   regression calibration workflow.
#'
#' @section Nuisance Covariates: Nuisance covariates are not of interest in the
#'   final model, but they must be accounted for when simulating long-term
#'   average intakes. For example, a dietary study with two recalls may want to
#'   account for whether a recall is a subject's second one and whether the
#'   recall was on a weekday or weekend. Nuisance covariates can be factored out
#'   by creating a row in `distrib.population` for each level of the covariate
#'   for every subject and creating a weighting variable with the weight for
#'   each level. An example of this procedure can be found in the daily nutrient
#'   analysis vignette.
#'
#' @section Dietary Supplements: Dietary supplements can be added to the usual
#'   intakes for one or more variables. This function uses the shrink-then-add
#'   method which means that the usual intake is calculated first and then the
#'   supplement is added. If a variable has a dietary supplement associated with
#'   it, a base usual intake and a supplemented intake are included in the
#'   output.
#'
#'
#' @param multivar.mcmc.model An `nci.multivar.mcmc` object.
#' @param distrib.population A data frame. Must contain all `covariates` and
#'   `never.consumer.covariates` in `multivar.mcmc.model`.
#' @param id Variable that identifies each subject in `distrib.population`.
#' @param weight Variable with weighting for each subject in
#'   `distrib.population`.
#' @param nuisance.weight Variable with the weighting for each nuisance variable
#'   level in `distrib.population`.
#' @param use.mcmc.u.matrices Flag specifying whether previously saved post-MCMC
#'   random effects ('u') matrix draws should be used instead of simulating new
#'   ones. Used in regression calibration procedures, see "Regression
#'   Calibration". (default = `FALSE`)
#' @param distrib.seed Integer starting seed for the random number generator. If
#'   `NULL`, uses a randomly generated integer from -10^7 to 10^7, exclusive.
#'   (default = `NULL`)
#' @param num.simulated.u Integer specifying the number of simulated random
#'   effect ('u') random draws should be made for each subject. Has no effect if
#'   `use.mcmc.u.matrices` is `TRUE`. (default = `500`)
#' @param dietary.supplements Named list of dietary supplement variable names.
#'   The names of the list are the variables which the supplements should be
#'   added to.
#' @param additional.output Vector of additional variables in
#'   `distrib.population` to be included in the output dataset.
#'
#' @returns An object of class `c("nci.multivar.distrib", "data.frame")` where
#'   each row represents a simulated subject. Contains the following columns:
#' * replicate: The sequence number for the simulated 'u' value used to simulate the intakes.
#' * `id`: Identifier for each subject.
#' * `weight`: Weighting for each subject.
#' * Variables specified in `additional.output`.
#' * `usual.intake` variables: The usual intake for each variable.
#' * `supplemented.intake` variables: The usual intake plus the supplement amount for variables with dietary supplements.
#' * `prob` variables: The long-term probability of consumption for episodic variables.
#' * `amount` variables: The long-term amount consumed given consumption for episodic variables.
#'
#'   The object also contains the following attribute:
#' * distrib.seed: The random number generator seed used to generate the results, see the `distrib.seed` parameter for details.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.sodium <- boxcox_survey(input.data=nhanes.subset,
#'                                row.subset=(nhanes.subset$DAY == 1),
#'                                variable="TSODI",
#'                                id="SEQN",
#'                                repeat.obs="DAY",
#'                                weight="WTDRD1",
#'                                covariates="RIDAGEYR")
#'
#' boxcox.g.whole <- boxcox_survey(input.data=nhanes.subset,
#'                                 row.subset=(nhanes.subset$DAY == 1),
#'                                 variable="G_WHOLE",
#'                                 is.episodic=TRUE,
#'                                 id="SEQN",
#'                                 repeat.obs="DAY",
#'                                 weight="WTDRD1",
#'                                 covariates="RIDAGEYR")
#'
#' boxcox.lambda.data <- rbind(boxcox.sodium, boxcox.g.whole)
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 episodic.variables="G_WHOLE",
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            episodic.variables="G_WHOLE",
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  episodic.variables="G_WHOLE",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' #use first instance of each subject as population base
#' mcmc.input.data <- pre.mcmc.data$mcmc.input
#' population.base <- mcmc.input.data[!duplicated(mcmc.input.data$SEQN),]
#'
#' distrib.output <- nci_multivar_distrib(multivar.mcmc.model=mcmc.output,
#'                                        distrib.population=population.base,
#'                                        id="SEQN",
#'                                        weight="WTDRD1",
#'                                        num.simulated.u=100,
#'                                        additional.output="RIDAGEYR")
#'
#' head(distrib.output)
nci_multivar_distrib <- function(multivar.mcmc.model,
                                 distrib.population,
                                 id,
                                 weight=NULL,
                                 nuisance.weight=NULL,
                                 use.mcmc.u.matrices=FALSE,
                                 distrib.seed=NULL,
                                 num.simulated.u=500,
                                 dietary.supplements=NULL,
                                 additional.output=NULL) {

  #1. Initialize parameters and data matrices for distrib main loop
  distrib.parameters <- initialize_distrib(distrib.population=distrib.population,
                                           id=id,
                                           nuisance.weight=nuisance.weight,
                                           mcmc.covariates=multivar.mcmc.model$covariates,
                                           mcmc.intercepts=multivar.mcmc.model$intercepts,
                                           beta=multivar.mcmc.model$beta,
                                           sigma.u=multivar.mcmc.model$sigma.u,
                                           sigma.e=multivar.mcmc.model$sigma.e,
                                           use.mcmc.u.matrices=use.mcmc.u.matrices,
                                           dietary.supplements=dietary.supplements,
                                           num.simulated.u=num.simulated.u,
                                           num.episodic=multivar.mcmc.model$num.episodic,
                                           num.daily=multivar.mcmc.model$num.daily,
                                           num.mcmc.iterations=multivar.mcmc.model$num.mcmc.iterations,
                                           num.burn=multivar.mcmc.model$num.burn,
                                           num.thin=multivar.mcmc.model$num.thin,
                                           num.post=multivar.mcmc.model$num.post,
                                           distrib.seed=distrib.seed,
                                           never.consumers.first.episodic=multivar.mcmc.model$never.consumers.first.episodic,
                                           never.consumer.covariates=multivar.mcmc.model$never.consumer.covariates,
                                           never.consumer.intercept=multivar.mcmc.model$never.consumer.intercept,
                                           alpha1=multivar.mcmc.model$alpha1)

  #2. Distrib main loop to create Monte Carlo dataset
  distrib.data <- distrib_main_loop(xbeta=distrib.parameters$xbeta,
                                    u.standard.deviation=distrib.parameters$u.standard.deviation,
                                    sigma.e.mean=distrib.parameters$sigma.e.mean,
                                    mcmc.u.matrices=multivar.mcmc.model$u.matrices.post,
                                    use.mcmc.u.matrices=use.mcmc.u.matrices,
                                    backtransformation.data=multivar.mcmc.model$backtransformation,
                                    records=distrib.parameters$records,
                                    subjects=distrib.parameters$subjects,
                                    subjects.mcmc=multivar.mcmc.model$mcmc.subjects,
                                    episodic.variables=multivar.mcmc.model$episodic.variables,
                                    daily.variables=multivar.mcmc.model$daily.variables,
                                    num.records=distrib.parameters$num.records,
                                    num.subjects=distrib.parameters$num.subjects,
                                    num.episodic=multivar.mcmc.model$num.episodic,
                                    num.daily=multivar.mcmc.model$num.daily,
                                    num.replicates=distrib.parameters$num.replicates,
                                    nuisance.weighting=distrib.parameters$nuisance.weighting,
                                    dietary.supplement.data=distrib.parameters$dietary.supplement.data,
                                    distrib.population=distrib.population,
                                    id=id,
                                    weight=weight,
                                    additional.output=additional.output,
                                    never.consumers.first.episodic=multivar.mcmc.model$never.consumers.first.episodic,
                                    consumer.probabilities=distrib.parameters$consumer.probabilities)

  #3. Output final dataset
  nci.multivar.distrib.data <- structure(distrib.data, class=c("nci.multivar.distrib", "data.frame"))
  attr(nci.multivar.distrib.data, "distrib.seed") <- distrib.parameters$distrib.seed
  return(nci.multivar.distrib.data)
}
