#' Fit MCMC measurement error model
#'
#' @description Fits a measurement error model for one or more episodic and/or
#'   daily variables using an MCMC method.
#'
#' @section About the model: The model fit by this function is a multivariate
#'   mixed-effects model with fixed effects ('beta') and a random intercept for
#'   each variable ('u'). Each daily variable has a single continuous outcome
#'   variable for the amount consumed. Each episodic variable has two model
#'   parts, a binary indicator of consumption and a continuous variable for
#'   amount consumed. A probit link function is used for modeling the binary
#'   indicator. The parts of the model (daily amounts, episodic indicators, and
#'   episodic amounts) can have separate covariate lists. The random intercepts
#'   are assumed to have multivariate Gaussian distribution with covariance
#'   matrix 'Sigma-u'. The residual error is also assumed to have a multivariate
#'   Gaussian distribution with covariance matrix 'Sigma-e'.
#'
#' @section MCMC procedure: The model is fit using a Gibbs sampler MCMC
#'   procedure. At each iteration, each parameter is updated sequentially using
#'   its distribution conditional on the other parameters. The conditional
#'   distributions and priors are outlined in Zhang, et al. (2011). After a
#'   specified burn-in period, subsequent samples of each parameter are averaged
#'   to calculate a posterior mean. The posterior means are the point estimates
#'   for the parameters. To ensure that the samples used to calculated the
#'   posterior mean are independent, the samples can be 'thinned'. For example,
#'   a thinning number of 5 means that every 5th sample after the burn-in period
#'   is used to calculate the posterior mean. When doing regression calibration,
#'   additional draws of the random intercept ('u') matrix can be made
#'   conditional on the posterior means of beta, Sigma-u, and Sigma-e. This is
#'   useful when performing measurement error correction with regression
#'   calibration.
#'
#' @section Never-consumers: The model can allow some subjects to be
#'   never-consumers of one episodic variable. If an episodic variable with
#'   never-consumers is specified, then the probability that each subject has
#'   any consumption is included as an additional part of the model for that
#'   variable. As with the other model parts, the never-consumer part of the
#'   model has its own covariate matrix and coefficients. To distiguish the
#'   coefficients from the regular fixed effect coefficients, they are referred
#'   to as 'alpha1'. Only one variable can be allowed to have never-consumers.
#'
#' @section Log-likelihood: An estimate of the log-likelihood marginalized over
#'   the random effects ('u') can be output after the MCMC iterations have
#'   finished. The random effects are integrated out using the Laplace
#'   approximation. The posterior mode of the random effects is found by the
#'   BFGS algorithm with derivatives approximated by the central difference
#'   method. Since the model is fit using an MCMC method, the marginal
#'   log-likelihood is never used in fitting the model. The log-likelihood
#'   output by this function corresponds well to the log-likelihood found by
#'   maximum likelihood for univariate and bivariate models. For
#'   higher-dimensional models and models allowing never-consumers that cannot
#'   be fit using maximum likelihood, the estimated likelihood is experimental
#'   and other measures of model fit should be considered. Estimating the
#'   log-likelihood increases computation time, especially as the number of
#'   variables increases.
#'
#' @section Regression Calibration: The model fit by `nci_multivar_mcmc()` can
#'   be used in regression calibration for measurement error correction. When
#'   doing regression calibration, random effect ('u') matrices must be drawn
#'   after the MCMC chain conditional on the posterior means of the parameters.
#'   This can be done by setting `num.post` to the number of post-MCMC 'u' draws
#'   to be made. Use \code{vignette("regression_calibration",
#'   package="ncimultivar")} for a full regression calibration workflow.
#'
#'
#'
#' @param pre.mcmc.data A list or `nci.multivar.preprocessor` object with the
#'   following elements:
#' * mcmc.input: A data frame with all variables needed for the MCMC model.
#' * backtransformation: A data frame with the following columns:
#'   * variable: The name of the variable.
#'   * tran_lambda: The value of the lambda used to transform the variable.
#'   * minamount: Half of the minimum non-zero value of the variable.
#'   * tran_center: The mean of the Box-Cox transformed variable before standardization.
#'   * tran_scale: The standard deviation of the Box-Cox transformed variable before standardization divided by `sqrt(2)`.
#'   * biomarker: Flag indicating whether the variable is a biomarker assumed to be unbiased on the transformed scale.
#' @param id Variable that identifies each subject.
#' @param weight Variable with weighting for each subject.
#' @param repeat.obs Variable that distinguishes repeat observations for each
#'   subject.
#' @param episodic.variables Vector of episodic variables.
#' @param episodic.indicators Vector of consumption indicators for episodic
#'   variables. Not needed if `pre.mcmc.data` is an`nci.multivar.preprocessor`
#'   object.
#' @param episodic.amounts Vector of consumption amount variables for episodic
#'   variables. Not needed if `pre.mcmc.data` is an `nci.multivar.preprocessor`
#'   object.
#' @param daily.variables Vector of daily variables.
#' @param daily.amounts Vector of consumption amount variables for daily
#'   variables. Not needed if `pre.mcmc.data` is an `nci.multivar.preprocessor`
#'   object.
#' @param default.covariates Vector of covariates to be used for episodic
#'   indicators, episodic amounts, and daily amounts. Does not affect the
#'   never-consumer model if present.
#' @param episodic.indicator.covariates Vector of covariates to be used for
#'   episodic indicators. If specified, overwrites `default.covariates` for
#'   episodic indicators.
#' @param episodic.amount.covariates Vector of covariates to be used for
#'   episodic amounts. If specified, overwrites `default.covariates` for
#'   episodic amounts.
#' @param daily.amount.covariates Vector of covariates to be used for daily
#'   amounts. If specified, overwrites `default.covariates` for daily amounts.
#' @param individual.covariates Named list of vectors of covariates to be used
#'   for individual indicators and amounts. Each name is the indicator or amount
#'   that the covariate list will be used for. Overrides `default.covariates`,
#'   `episodic.indicator.covariates`, `episodic.amount.covariates`, and
#'   `daily.amount.covariates` for each named variable.
#' @param default.intercept Flag to include an intercept in the models for
#'   episodic indicators, episodic amounts, and daily amounts. Does not affect
#'   the never-consumer model if present. (default = `TRUE`)
#' @param episodic.indicator.intercept Flag to include an intercept in the model
#'   for episodic indicators. If specified, overwrites `default.intercept` for
#'   episodic indicators.
#' @param episodic.amount.intercept Flag to include an intercept in the model
#'   for episodic amounts. If specified, overwrites `default.intercept` for
#'   episodic amounts.
#' @param daily.amount.intercept Flag to include an intercept in the model for
#'   daily amounts. If specified, overwrites `default.intercept` for daily
#'   amounts.
#' @param individual.intercept Named list of flags to include intercepts in the
#'   model for individual indicators and amounts. Each name is the indicator or
#'   amount that the intercept will be used for. Overrides `default.intercept`,
#'   `episodic.indicator.intercept`, `episodic.amount.intercept`, and
#'   `daily.amount.intercept` for each named variable.
#' @param never.consumer.variable One episodic variable to allow
#'   never-consumers. Can be a variable already listed as episodic or a new
#'   variable.
#' @param never.consumer.indicator Consumption indicator variable for the
#'   never-consumer variable. Not needed if `pre.mcmc.data` is an
#'   `nci.multivar.preprocessor` object.
#' @param never.consumer.amount Consumption amount variable for the
#'   never-consumer variable. Not needed if `pre.mcmc.data` is an
#'   `nci.multivar.preprocessor` object.
#' @param never.consumer.covariates Vector of covariates to be used for the
#'   never-consumer model. Not affected by `default.covariates` and must be
#'   specified separately.
#' @param never.consumer.intercept Flag to include intercept in the
#'   never-consumer model. Not affected by `default.intercept` and must be
#'   specified separately. (default = `TRUE`)
#' @param mcmc.seed Integer starting seed for the random number generator. If
#'   `NULL`, uses a randomly generated integer from -10^7 to 10^7, exclusive.
#'   (default = `NULL`)
#' @param num.mcmc.iterations Integer specifying the total number (including
#'   burn-in) of iterations in the MCMC chain. (default = `12000`)
#' @param num.burn Integer specifying the number of burn-in iterations in the
#'   MCMC chain. Must be smaller than `num.mcmc.iterations`. (default = `2000`)
#' @param num.post Integer specifying the number of random effect ('u') matrix
#'   draws done after the MCMC chain is finished, conditional on the posterior
#'   mean of the parameters. Used for regression calibration, see "Regression
#'   Calibration". (default = `0`)
#' @param num.thin Integer specifying the number of iterations between MCMC
#'   samples used for calculating posterior means. (default = `25`)
#' @param sigma.u.prior Matrix specifying a prior covariance matrix for the
#'   random effects ('u'). Must be a valid covariance matrix with dimensions
#'   equal to the total number of episodic indicators, episodic amounts, and
#'   daily amounts.
#' @param sigma.u.constant Flag specifying if the covariance matrix of the
#'   random effects ('u') should remain the same for each iteration (default =
#'   `FALSE`).
#' @param save.u.main Flag specifying if random effects ('u') matrices should be
#'   saved from the main MCMC chain. If `TRUE`, saves 'u' matrix every
#'   `num.thin` iterations after `num.burn`. This is used for fitting models
#'   with techniques such as multiple imputation. Does not affect conditional
#'   'u' matrices drawn after the MCMC chain. (default = `FALSE`)
#' @param save.all.u Flag specifying if random effects ('u') matrices are saved
#'   for every iteration. Greatly increases disk space needed to store output
#'   object. Has no effect if `save.u.main` is FALSE. (default = `FALSE`)
#' @param do.log.likelihood Flag specifying if an estimate of the log-likelihood
#'   should be calculated after the MCMC chain has finished. See the
#'   'Log-likelihood' section for details. (default = `FALSE`)
#' @param log.file.name Name of the file to log notes, warnings, and errors. If
#'   `NULL`, no log file will be output. (default = `NULL`)
#' @param log.threshold Integer specifying which messages to output to the log:
#' * 1: All notes, warnings, and errors (default)
#' * 2: Only warnings and errors
#' * 3: Only errors
#' @param log.to.console Flag specifying whether log messages should be printed
#'   to the R console in addition to the log file (default = `TRUE`).
#'
#' @returns An `nci.multivar.mcmc` object with the following elements:
#' * log.likelihood: The estimated marginal log-likelihood of the model, `NA` if `do.log.likelihood` is FALSE.
#' * beta: Fixed effect coefficients at every MCMC iteration.
#' * sigma.u: Random effect covariance matrix at every MCMC iteration.
#' * sigma.e: Residual error covariance matrix at every MCMC iteration.
#' * alpha1: Coefficients of never-consumer model at every MCMC iteration, NULL if no variables allow never-consumers.
#' * consumer.probabilities: Probability that each subject is a consumer of the never-variable at every MCMC iteration, NULL if no variables allow never-consumers.
#' * u.matrices.main: Random effect ('u') matrices saved from the main MCMC chain.
#' * u.matrices.post: Random effect ('u') matrices drawn after the MCMC chain conditional on the posterior means of the parameters.
#' * mcmc.subjects: Vector of subject IDs for each unique subject in the data.
#' * subject.weighting: Vector of Weighting of each unique subject in the data.
#' * num.episodic: Number of episodic variables in the model.
#' * num.daily: Number of daily variables in the model.
#' * num.trace: Number of main chain MCMC traces saved.
#' * num.mcmc.iterations: Total number of iterations in the MCMC chain including burn-in, same as `num.mcmc.iterations` parameter.
#' * num.burn: Number of burn-in iterations that will be discarded when calculating posterior means, same as `num.burn` parameter.
#' * num.thin: Number of iterations between MCMC samples used for calculating posterior means, same as `num.thin` parameter.
#' * num.post: Number of 'u' matrices drawn after the MCMC chain conditional on the posterior means of the parameters, same as `num.post` parameter.
#' * saved.u.main: Numeric vector of MCMC iterations that have saved 'u' matrices, `NULL` if no main MCMC chain 'u' matrices are saved.
#' * backtransformation: A data frame with metadata for variable backtransformation, same as `backtransformation` element of the `pre.mcmc.data` parameter.
#' * episodic.variables: Episodic variables used in the model. The never-consumer variable (if any) is always first.
#' * episodic.indicators: Indicators for episodic variables in the model. The indicator for the never-consumer variable (if any) is always first.
#' * episodic.amounts: Standardized consumption amounts for episodic variables in the model. The amount for the never-consumer variable (if any) is always first.
#' * daily.variables: Daily variables used in the model.
#' * daily.amounts: Standardized consumption amounts for daily variables in the model.
#' * covariates: List of vectors of names of covariates for each indicator and amount.
#' * intercepts: List of flags for whether an intercept is included for each indicator and amount.
#' * never.consumers.first.episodic: Flag for whether a variable with never-consumers allowed was specified.
#' * never.consumer.covariates: Vector of names of covariates for never-consumer model, `NULL` if never-consumers are not present.
#' * never.consumer.intercept: Flag for whether an intercept is included in the never-consumer model, `NULL` if never-consumers are not present.
#'
#'   The object also contains the following attribute:
#' * mcmc.seed: The random number generator seed used to generate the results, see the `mcmc.seed` parameter for details.
#'
#' @export
#'
#' @examples
#' #subset NHANES data
#' nhanes.subset <- nhcvd[nhcvd$SDMVSTRA %in% c(48, 60, 72),]
#'
#' boxcox.sodium <- boxcox_survey(input.data=nhanes.subset,
#'                                row.subset=(nhanes.subset$DAY == 1),
#'                                variable="TSODI",
#'                                id="SEQN",
#'                                repeat.obs="DAY",
#'                                weight="WTDRD1",
#'                                covariates="RIDAGEYR")
#'
#' boxcox.g.whole <- boxcox_survey(input.data=nhanes.subset,
#'                                 row.subset=(nhanes.subset$DAY == 1),
#'                                 variable="G_WHOLE",
#'                                 is.episodic=TRUE,
#'                                 id="SEQN",
#'                                 repeat.obs="DAY",
#'                                 weight="WTDRD1",
#'                                 covariates="RIDAGEYR")
#'
#' boxcox.lambda.data <- rbind(boxcox.sodium, boxcox.g.whole)
#'
#' minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
#'                                                 row.subset=(nhanes.subset$DAY == 1),
#'                                                 episodic.variables="G_WHOLE",
#'                                                 daily.variables="TSODI")
#'
#' pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
#'                                            episodic.variables="G_WHOLE",
#'                                            daily.variables="TSODI",
#'                                            continuous.covariates="RIDAGEYR",
#'                                            boxcox.lambda.data=boxcox.lambda.data,
#'                                            minimum.amount.data=minimum.amount.data)
#'
#' mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
#'                                  id="SEQN",
#'                                  weight="WTDRD1",
#'                                  repeat.obs="DAY",
#'                                  episodic.variables="G_WHOLE",
#'                                  daily.variables="TSODI",
#'                                  default.covariates="std.RIDAGEYR",
#'                                  num.mcmc.iterations=1000,
#'                                  num.burn=500,
#'                                  num.thin=1)
#'
#' names(mcmc.output)
nci_multivar_mcmc <- function(pre.mcmc.data,
                              id,
                              weight=NULL,
                              repeat.obs,
                              episodic.variables=NULL,
                              episodic.indicators=NULL,
                              episodic.amounts=NULL,
                              daily.variables=NULL,
                              daily.amounts=NULL,
                              default.covariates=NULL,
                              episodic.indicator.covariates=NULL,
                              episodic.amount.covariates=NULL,
                              daily.amount.covariates=NULL,
                              individual.covariates=list(),
                              default.intercept=TRUE,
                              episodic.indicator.intercept=NULL,
                              episodic.amount.intercept=NULL,
                              daily.amount.intercept=NULL,
                              individual.intercept=list(),
                              never.consumer.variable=NULL,
                              never.consumer.indicator=NULL,
                              never.consumer.amount=NULL,
                              never.consumer.covariates=NULL,
                              never.consumer.intercept=TRUE,
                              mcmc.seed=NULL,
                              num.mcmc.iterations=12000,
                              num.burn=2000,
                              num.post=0,
                              num.thin=25,
                              sigma.u.prior=NULL,
                              sigma.u.constant=FALSE,
                              save.u.main=FALSE,
                              save.all.u=FALSE,
                              do.log.likelihood=FALSE,
                              log.file.name=NULL,
                              log.threshold=1,
                              log.to.console=TRUE) {

  logger <- create_logger(log.file=log.file.name, log.to.console=log.to.console, log.threshold=log.threshold)

  #1. Check input parameters
  logger <- check_input(logger=logger,
                        num.mcmc.iterations=num.mcmc.iterations,
                        num.burn=num.burn,
                        num.thin=num.thin,
                        num.post=num.post,
                        pre.mcmc.data=pre.mcmc.data,
                        id=id,
                        repeat.obs=repeat.obs,
                        weight=weight,
                        episodic.variables=episodic.variables,
                        episodic.indicators=episodic.indicators,
                        episodic.amounts=episodic.amounts,
                        daily.variables=daily.variables,
                        daily.amounts=daily.amounts,
                        never.consumer.variable=never.consumer.variable,
                        never.consumer.indicator=never.consumer.indicator,
                        never.consumer.amount=never.consumer.amount,
                        default.covariates=default.covariates,
                        episodic.indicator.covariates=episodic.indicator.covariates,
                        episodic.amount.covariates=episodic.amount.covariates,
                        daily.amount.covariates=daily.amount.covariates,
                        individual.covariates=individual.covariates,
                        never.consumer.covariates=never.consumer.covariates,
                        sigma.u.prior=sigma.u.prior)

  #2. Process variable lists
  variable.list <- process_variable_lists(pre.mcmc.data=pre.mcmc.data,
                                          episodic.variables=episodic.variables,
                                          episodic.indicators=episodic.indicators,
                                          episodic.amounts=episodic.amounts,
                                          daily.variables=daily.variables,
                                          daily.amounts=daily.amounts,
                                          never.consumer.variable=never.consumer.variable,
                                          never.consumer.indicator=never.consumer.indicator,
                                          never.consumer.amount=never.consumer.amount)

  #3. Process covariate lists
  covariate.list <- process_covariate_lists(pre.mcmc.data=pre.mcmc.data,
                                            default.covariates=default.covariates,
                                            episodic.indicator.covariates=episodic.indicator.covariates,
                                            episodic.amount.covariates=episodic.amount.covariates,
                                            daily.amount.covariates=daily.amount.covariates,
                                            individual.covariates=individual.covariates,
                                            default.intercept=default.intercept,
                                            episodic.indicator.intercept=episodic.indicator.intercept,
                                            episodic.amount.intercept=episodic.amount.intercept,
                                            daily.amount.intercept=daily.amount.intercept,
                                            individual.intercept=individual.intercept,
                                            episodic.indicators=variable.list$episodic.indicator,
                                            episodic.amounts=variable.list$episodic.amount,
                                            daily.amounts=variable.list$daily.amount,
                                            never.consumers.first.episodic=variable.list$never.consumers.first.episodic,
                                            never.consumer.covariates=never.consumer.covariates,
                                            never.consumer.intercept=never.consumer.intercept)

  #4. Initializing parameters for main loop: sets seed, population-level variables, data matrices, covariate matrices, and priors
  mcmc.parameters <- initialize_mcmc(pre.mcmc.data=pre.mcmc.data,
                                     id=id,
                                     repeat.obs=repeat.obs,
                                     weight=weight,
                                     variable.list=variable.list,
                                     covariate.list=covariate.list,
                                     mcmc.seed=mcmc.seed,
                                     sigma.u.prior=sigma.u.prior)

  #5. Pre-loop error checking of initialized parameters
  logger <- check_initialized(logger=logger,
                              covariate.matrices=mcmc.parameters$covariate.matrices,
                              mcmc.priors=mcmc.parameters$priors,
                              save.u.main=save.u.main,
                              save.all.u=save.all.u,
                              num.mcmc.iterations=num.mcmc.iterations,
                              num.burn=num.burn,
                              num.thin=num.thin,
                              num.post=num.post,
                              never.consumers.first.episodic=variable.list$never.consumers.first.episodic)


  #6. Main MCMC loop
  mcmc.output <- mcmc_main_loop(num.mcmc.iterations=num.mcmc.iterations,
                                num.burn=num.burn,
                                num.post=num.post,
                                num.thin=num.thin,
                                num.subjects=mcmc.parameters$subject.data$num.subjects,
                                num.episodic=mcmc.parameters$subject.data$num.episodic,
                                num.daily=mcmc.parameters$subject.data$num.daily,
                                num.recalls=mcmc.parameters$subject.data$num.recalls,
                                recall.availability=mcmc.parameters$subject.data$recall.availability,
                                subject.weighting=mcmc.parameters$subject.data$weighting,
                                episodic.indicator.matrices=mcmc.parameters$variable.matrices$episodic.indicators,
                                never.consumers.first.episodic=variable.list$never.consumers.first.episodic,
                                covariate.matrices=mcmc.parameters$covariate.matrices$recall,
                                weighted.covariate.matrices=mcmc.parameters$covariate.matrices$weighted.recall,
                                weighted.covariate.squared.sums=mcmc.parameters$covariate.matrices$weighted.covariate.squared.sums,
                                never.consumer.covariate.matrix=mcmc.parameters$covariate.matrices$never.consumer,
                                alpha1.mean.prior=mcmc.parameters$priors$alpha1.prior$mean,
                                alpha1.covariance.prior=mcmc.parameters$priors$alpha1.prior$covariance,
                                consumer.probabilities.prior=mcmc.parameters$priors$consumer.probabilities.prior,
                                beta.mean.prior=mcmc.parameters$priors$beta.prior$mean,
                                beta.covariance.prior=mcmc.parameters$priors$beta.prior$covariance,
                                r.matrix.prior=mcmc.parameters$priors$r.matrix.prior,
                                theta.matrix.prior=mcmc.parameters$priors$theta.matrix.prior,
                                v.matrix.prior=mcmc.parameters$priors$v.matrix.prior,
                                sigma.e.prior=mcmc.parameters$priors$sigma.e.prior,
                                sigma.u.prior=mcmc.parameters$priors$sigma.u.prior,
                                u.matrix.prior=mcmc.parameters$priors$u.matrix.prior,
                                w.matrix.prior=mcmc.parameters$priors$w.matrix.prior,
                                xbeta.prior=mcmc.parameters$priors$xbeta.prior,
                                xbeta.u.prior=mcmc.parameters$priors$xbeta.u.prior,
                                sigma.u.constant=sigma.u.constant,
                                save.u.main=save.u.main,
                                save.all.u=save.all.u,
                                do.log.likelihood=do.log.likelihood)

  #7. Output: Returns MCMC_multivar object
  mcmc.data <- list(log.likelihood=mcmc.output$log.likelihood,
                    beta=mcmc.output$beta,
                    sigma.u=mcmc.output$sigma.u,
                    sigma.e=mcmc.output$sigma.e,
                    alpha1=mcmc.output$alpha1,
                    consumer.probabilities=mcmc.output$consumer.probabilities,
                    u.matrices.main=mcmc.output$u.matrices.main,
                    u.matrices.post=mcmc.output$u.matrices.post,
                    mcmc.subjects=unique(mcmc.parameters$subject.data$subjects),
                    subject.weighting=mcmc.parameters$subject.data$weighting,
                    num.episodic=mcmc.parameters$subject.data$num.episodic,
                    num.daily=mcmc.parameters$subject.data$num.daily,
                    num.trace=num.mcmc.iterations,
                    num.mcmc.iterations=num.mcmc.iterations,
                    num.burn=num.burn,
                    num.thin=num.thin,
                    num.post=num.post,
                    saved.u.main=mcmc.output$saved.u.main,
                    backtransformation=pre.mcmc.data$backtransformation,
                    episodic.variables=variable.list$episodic.variables,
                    episodic.indicators=variable.list$episodic.indicators,
                    episodic.amounts=variable.list$episodic.amounts,
                    daily.variables=variable.list$daily.variables,
                    daily.amounts=variable.list$daily.amounts,
                    covariates=covariate.list$variables$covariates,
                    intercepts=covariate.list$variables$intercepts,
                    never.consumers.first.episodic=variable.list$never.consumers.first.episodic,
                    never.consumer.covariates=covariate.list$never.consumer$covariates,
                    never.consumer.intercept=covariate.list$never.consumer$intercept)

  nci.multivar.mcmc.data <- structure(mcmc.data, class="nci.multivar.mcmc")
  attr(nci.multivar.mcmc.data, "mcmc.seed") <- mcmc.parameters$mcmc.seed

  return(nci.multivar.mcmc.data)
}
