#' Fix NHANES Strata for BRR
#'
#' @description Corrects specific NHANES strata that do not have exactly 2
#'   primary sampling units (PSUs) so that balanced repeated replication (BRR)
#'   weights can be generated for the data.
#'
#' @details Some NHANES cycles have strata that contain 3 primary sampling units
#'   (PSUs). Since balanced repeated replication (BRR) requires exactly 2 PSUs
#'   for all strata, PSUs in these strata must be reconfigured for BRR to be
#'   used. This function corrects specific NHANES strata that have 3 PSUs. Two
#'   of the three PSUs in each of the strata to be corrected were randomly
#'   selected to be combined. the If the function detects other strata that it
#'   was not able to correct, a warning is given to the user.
#'
#'   This function currently corrects the following strata/PSU combinations:
#'   * SDMVSTRA = 1, Combine SDMVPSU 1 and 2
#'   * SDMVSTRA = 86, Combine SDMVPSU 2 and 3
#'   * SMDVSTRA = 90, Combine SDMVPSU 2 and 3
#'   * SMDVSTRA = 91, Combine SDMVPSU 1 and 3
#'   * SDMVSTRA = 92, Combine SDMVPSU 2 and 3
#'   * SDMVSTRA = 156, Combine SDMVPSU 1 and 3
#'
#'
#' @param input.data A data.frame with NHANES data. It must contain the
#'   following variables:
#'   * SDMVSTRA: Strata
#'   * SDMVPSU: Primary Sampling Unit (PSU)
#'
#' @returns A data.frame identical to `input.data` with `SDMVPSU` modified so
#'   that selected strata have exactly 2 PSUs.
#'
#' @export
#'
#' @examples
#' nhanes.data <- fix_nhanes_strata(nhcvd)
fix_nhanes_strata <- function(input.data) {

  #Fix known problematic NHANES strata that do not have exactly 2 PSUs
  strata.psu.levels <- unique(input.data[,c("SDMVSTRA", "SDMVPSU")])

  if(any(strata.psu.levels$SDMVSTRA == 1 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 1 & input.data$SDMVPSU == 2] <- 1
    input.data$SDMVPSU[input.data$SDMVSTRA == 1 & input.data$SDMVPSU == 3] <- 2
  }

  if(any(strata.psu.levels$SDMVSTRA == 86 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 86 & input.data$SDMVPSU == 3] <- 2
  }

  if(any(strata.psu.levels$SDMVSTRA == 90 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 90 & input.data$SDMVPSU == 3] <- 2
  }

  if(any(strata.psu.levels$SDMVSTRA == 91 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 91 & input.data$SDMVPSU == 3] <- 1
  }

  if(any(strata.psu.levels$SDMVSTRA == 92 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 92 & input.data$SDMVPSU == 3] <- 2
  }

  if(any(strata.psu.levels$SDMVSTRA == 156 & strata.psu.levels$SDMVPSU == 3)) {

    input.data$SDMVPSU[input.data$SDMVSTRA == 156 & input.data$SDMVPSU == 3] <- 1
  }

  #Check if any more strata do not have exactly 2 PSUs
  psu.levels <- unique(input.data[,c("SDMVSTRA", "SDMVPSU")])
  psu.counts <- aggregate(psu.levels[,"SDMVPSU",drop=FALSE],
                          by=psu.levels[,"SDMVSTRA",drop=FALSE],
                          length)

  if(any(psu.counts$SDMVPSU != 2)) {

    not.fixed <- psu.counts$SDMVSTRA[psu.counts$SDMVPSU != 2]
    warning(paste0("The following strata do not have exactly 2 PSUs but could not be fixed: ", paste(not.fixed, collapse=", ")))
  }

  return(input.data)
}
