% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary_functions.R
\name{summary_coefficients}
\alias{summary_coefficients}
\alias{summary_coefficients.mlm}
\alias{summary_coefficients.glm}
\alias{summary_coefficients.lm}
\title{Summarize coefficients of a regression model}
\usage{
summary_coefficients(model, population.name = "All")

\method{summary_coefficients}{mlm}(model, population.name = "All")

\method{summary_coefficients}{glm}(model, population.name = "All")

\method{summary_coefficients}{lm}(model, population.name = "All")
}
\arguments{
\item{model}{A model object. See Details for compatible object classes.}

\item{population.name}{A character string to identify a population. Included
as a column in the output so that populations can be distinguished when
output datasets are combined. (default = \code{"All"})}
}
\value{
A data frame with the following columns:
\itemize{
\item population: The name of the population given by \code{population.name}.
\item variable: The name of the response variable.
\item statistic: A character string with the name of the covariate.
\item value: The value of the model coefficient.
}
}
\description{
A utility that extracts the coefficients of a model object and
puts them into a data frame. The output data frame is structured so that
the summary statistic values are in a single column to make variance
calculation from replicate methods more efficient (for an example, see the
regression calibration vignette).
}
\details{
This function is compatible with models with class \code{lm}, \code{glm}, or
\code{mlm}. This includes models output by \code{\link[stats:lm]{stats::lm()}} and \code{\link[stats:glm]{stats::glm()}} as
well as extensions of those models such as from the 'survey' and 'mfp'
packages.
}
\examples{
#subset NHANES data
nhanes.subset <- nhcvd[nhcvd$SDMVSTRA \%in\% c(48, 60, 72),]

#Use NCI method to simulate usual intakes of potassium
boxcox.lambda.data <- boxcox_survey(input.data=nhanes.subset,
                                    row.subset=(nhanes.subset$DAY == 1),
                                    variable="TPOTA",
                                    id="SEQN",
                                    repeat.obs="DAY",
                                    weight="WTDRD1")

minimum.amount.data <- calculate_minimum_amount(input.data=nhanes.subset,
                                                row.subset=(nhanes.subset$DAY == 1),
                                                daily.variables="TPOTA")

pre.mcmc.data <- nci_multivar_preprocessor(input.data=nhanes.subset,
                                           daily.variables="TPOTA",
                                           boxcox.lambda.data=boxcox.lambda.data,
                                           minimum.amount.data=minimum.amount.data)

#for regression calibration, post-MCMC random effect ('u') draws are saved
mcmc.output <- nci_multivar_mcmc(pre.mcmc.data=pre.mcmc.data,
                                 id="SEQN",
                                 weight="WTDRD1",
                                 repeat.obs="DAY",
                                 daily.variables="TPOTA",
                                 num.mcmc.iterations=1000,
                                 num.burn=500,
                                 num.thin=1,
                                 num.post=250)

mcmc.input.data <- pre.mcmc.data$mcmc.input
population.base <- mcmc.input.data[!duplicated(mcmc.input.data$SEQN),]

#use post-MCMC random effect ('u') draws instead of simulating new ones
distrib.output <- nci_multivar_distrib(multivar.mcmc.model=mcmc.output,
                                       distrib.population=population.base,
                                       id="SEQN",
                                       weight="WTDRD1",
                                       use.mcmc.u.matrices=TRUE,
                                       additional.output="BPSY_AVG")

#calculate average usual intake per subject
regression.data <- aggregate(distrib.output[,"usual.intake.TPOTA",drop=FALSE],
                             by=distrib.output[,c("SEQN", "BPSY_AVG"),drop=FALSE],
                             mean)


#convert usual intake of potassium from milligrams to grams
regression.data$usual.intake.TPOTA <- regression.data$usual.intake.TPOTA/1000

#fit model of systolic blood pressure as a function of usual potassium intake in grams
blood.pressure <- lm(BPSY_AVG ~ usual.intake.TPOTA, data=regression.data)

#summary of blood pressure model coefficients
blood.pressure.summary <- summary_coefficients(model=blood.pressure)
blood.pressure.summary
}
