/*
Program Purpose:
IML modules for distrib main loop
*/

libname _modules ".";

proc iml;

	**Produces simulated Monte Carlo datasets using MCMC multivar parameters;
	start distrib_main_loop(xbeta,
													u_standard_deviation,
													sigma_e_mean,
													mcmc_u_matrices,
													use_mcmc_u_matrices,
													backtransformation_data,
													records,
													subjects,
													subjects_mcmc,
													episodic_variables,
													daily_variables,
													num_records,
													num_subjects,
													num_episodic,
													num_daily,
													num_replicates,
													nuisance_weighting,
													variables_to_supplement,
													dietary_supplement_data,
													distrib_population,
													id,
													weight,
													additional_output,
													has_never_consumers,
													consumer_probabilities,
													outname);
		
		**If MCMC U matrices are used, determine which rows of U matrices come from MCMC U matrices and which must be simulated;
		mcmc_u_index = {};
		mcmc_distrib_match = {};
		sim_u_index = {};
		if upcase(use_mcmc_u_matrices) = "Y" then do;
			
			if any(element(subjects, subjects_mcmc)) then do;
			
				mcmc_u_index = loc(element(subjects, subjects_mcmc));
				mcmc_distrib_match = match(subjects[mcmc_u_index], subjects_mcmc);
			end;
			
			if any(^element(subjects, subjects_mcmc)) then do;
			
				sim_u_index = loc(^element(subjects, subjects_mcmc));
			end;
		end;
		
		**match records in distrib to subjects;
		subject_record_match = match(records, subjects);

		**main loop of replicates;
		mcmc_u_matrix = {};
		do replicate = 1 to num_replicates;
		
			**generate U matrix from MCMC output and simulated data (if needed);
			if upcase(use_mcmc_u_matrices) = "Y" then do;
			
				mcmc_u_matrix = mcmc_u_matrices$replicate;
			end;
			
			u_matrix = generate_u_matrix(use_mcmc_u_matrices,
																	 mcmc_u_matrix,
																	 u_standard_deviation,
																	 mcmc_u_index,
																	 mcmc_distrib_match,
																	 sim_u_index,
																	 num_subjects,
																	 num_episodic,
																	 num_daily);
																	 
			**calculate XBeta + U;
			xbeta_u = xbeta + u_matrix[subject_record_match,];
			
			**find subjects that are never-consumers of the first episodic food;
			never_consumers = find_never_consumers(has_never_consumers,
																					 	 consumer_probabilities,
																					 	 subject_record_match,
																					 	 num_subjects);
																								 
			**calculate consumption probabilities of each episodic food for each subject;
			call calculate_probability(consumption_probability,
																 probability_names,
																 xbeta_u,
																 episodic_variables,
																 num_records,
																 num_episodic,
																 has_never_consumers,
																 never_consumers);
			
			**calculate backtransformed amount of each episodic food for each subject;
			call calculate_amount(backtransformed_amount,
														amount_names,
														xbeta_u,
														sigma_e_mean,
														backtransformation_data,
														episodic_variables,
														daily_variables,
														num_records,
														num_episodic,
														num_daily,
														has_never_consumers,
														never_consumers);
			
			**calculate usual intake (t values);
			call calculate_usual_intake(usual_intake,
																	usual_intake_names,
																	consumption_probability,
																	backtransformed_amount,
																	episodic_variables,
																	daily_variables,
																	num_records,
																	num_episodic,
																	num_daily,
																	has_never_consumers,
																	never_consumers);
			
			**calculate supplemented intake;
			call calculate_supplemented_intake(supplemented_intake,
																				 supplemented_names,
																				 usual_intake,
																				 dietary_supplement_data,
																				 variables_to_supplement,
																				 episodic_variables,
																				 daily_variables,
																				 num_records);
			
			**combine intakes, probabilities, and amounts into dataset;
			distrib_replicate = usual_intake;
			distrib_names = usual_intake_names;
			
			if ncol(variables_to_supplement) > 0 then do;
			
				distrib_replicate = distrib_replicate || supplemented_intake;
				distrib_names = distrib_names || supplemented_names;
			end;
			
			if num_episodic > 0 then do;
			
				distrib_replicate = distrib_replicate || consumption_probability || backtransformed_amount[,1:num_episodic];
				distrib_names = distrib_names || probability_names || amount_names[,1:num_episodic];
			end;
			
			create _distrib_replicate from records nuisance_weighting distrib_replicate[colname=(id || {"_nuisance_weight"} || distrib_names)];
				append from records nuisance_weighting distrib_replicate;
			close _distrib_replicate;
			
			submit id weight additional_output distrib_names replicate distrib_population;
				**sum over nuisance variable levels;
				proc means data=_distrib_replicate noprint;
					by &id;
					
					var &distrib_names;
					weight _nuisance_weight;
					
					output out=_distrib&replicate (keep = &id &distrib_names) 
								 sum(&distrib_names)=&distrib_names;
				run;
				
				**merge in subject weights and additional variables;
				data _distrib&replicate;
					merge &distrib_population (keep = &id &weight &additional_output)
								_distrib&replicate;
					by &id;
					if first.&id;
				run;
				
				data _distrib&replicate;
					set _distrib&replicate;
					
					replicate = &replicate;
				run;
			endsubmit;
		end;
		
		**output replicates;
		submit outname num_replicates;
			data &outname;
				set _distrib1-_distrib&num_replicates;
			run;
		endsubmit;
		
		**clean up datasets;
		call delete("_distrib_replicate");
		call delete(cat("_distrib", 1:num_replicates));
	finish;
	
	reset storage=_modules.distrib_modules;
	store module=(distrib_main_loop);
quit;