/*
Extract NCI Multivar MCMC Parameters

Description:
	
	Extract posterior means and covariances from an NCI multivar MCMC model.
	
Parameters:

	- multivar_mcmc_lib: The SAS library that contains the output datasets of an MCMC model. (default = WORK)
	- multivar_mcmc_model: Prefix for the output datasets of the MCMC model (max 23 characters).
												 See nci_multivar_mcmc macro for more details.
	- outlib: The SAS library to store output datasets. (default = WORK)
	- outname: The prefix used to name output datasets (max 26 characters).
	
Output:

	The following SAS data sets are created in outlib and prefixed with outname:
	
		- _means: A SAS data set of posterior means of MCMC parameters.
		- _cov: A SAS data set of posterior covariances of MCMC parameters.
								 
Details:

	This function calculates the posterior means and covariances of
	parameters using the MCMC samples in an NCI multivar MCMC model. Means
	and covariances are calculated for the fixed effect coefficients (beta),
	random effect covariance (Sigma-u), and residual error covariance
	(Sigma-e) for all models. For models allowing never-consumers, the
	never-consumer model coefficients (alpha1) and mean consumer probability
	are also included. The calculations use samples after the burn-in period,
	spaced apart by the thinning number. It is important to note that the
	covariance matrix returned by this function is the covariance matrix of the
	Monte Carlo process, and it is distinct from the covariances that would be
	found from replication techniques such as bootstrap or BRR.
*/
%macro extract_parameters(multivar_mcmc_lib=WORK,
													multivar_mcmc_model=,
													outlib=WORK,
													outname=);
													
	%local num_variables var_j
				 has_never_consumers
				 num_mcmc_iterations num_burn num_thin;

	**number of iterations and burn-in;
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._iters;
		
		call symputx("num_mcmc_iterations", num_mcmc_iterations);
		call symputx("num_burn", num_burn);
		call symputx("num_thin", num_thin);
	run;
	
	**number of variables in the model;
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
		
		num_variables = 2*num_episodic + num_daily;
		call symputx("num_variables", num_variables);
		
		call symputx("has_never_consumers", has_never_consumers);
	run;
	
	**extract samples of MCMC parameters with burn-in and thinned iterations discarded;
	data _mcmc_samples;
		%do var_j = 1 %to &num_variables.;
			set &multivar_mcmc_lib..&multivar_mcmc_model._beta&var_j.;
		%end;
		set &multivar_mcmc_lib..&multivar_mcmc_model._sigma_e;
		set &multivar_mcmc_lib..&multivar_mcmc_model._sigma_u;
		%if &has_never_consumers. = 1 %then %do;
			set &multivar_mcmc_lib..&multivar_mcmc_model._alpha1;
			set &multivar_mcmc_lib..&multivar_mcmc_model._conprob1;
		%end;
		if _N_ > &num_burn. and 
			 _N_ <= &num_mcmc_iterations. and 
			 mod(_N_, &num_thin.) = mod(&num_burn. + 1, &num_thin.);
	run;
	
	**Calculate mean and covariance matrix;
	proc corr data=_mcmc_samples outp=_outcorr cov noprint;
		var _all_;
	run;
	
	data &outlib..&outname._means (keep = beta: sigma_e: sigma_u:
																						 %if &has_never_consumers. = 1 %then %do;
																						 	 alpha1: consumer_probability1
																						 %end;
																		 );
		set _outcorr;
		if _TYPE_ = "MEAN";
		
		_NAME_ = "All";
	run;
	
	data &outlib..&outname._cov (keep = _NAME_
																					 beta: sigma_e: sigma_u:
																					 %if &has_never_consumers. = 1 %then %do;
																					 	 alpha1: consumer_probability1
																					 %end;
																	 );
		set _outcorr;
		if _TYPE_ = "COV";
	run;
%mend extract_parameters;

/*
Modify Burn-In and Thinning for MCMC Model

Description:

	This macro modifies the burn-in and thinning values of a
	multivar MCMC model that are used to calculate posterior means and
	covariances. This can be useful for determining the impact of different
	values and finding optimal numbers that balance accuracy and computation
	speed.
	
Parameters:

	- multivar_mcmc_lib: The SAS library that contains the output datasets of an MCMC model. (default = WORK)
	- multivar_mcmc_model: Prefix for the output datasets of the MCMC model (max 23 characters).
												 See nci_multivar_mcmc macro for more details.
	- num_mcmc_iterations: An integer specifying the new total number of iterations in the MCMC chain, including burn-in.
	- num_burn: An integer specifying the new number of burn-in iterations in the MCMC chain.
	- num_thin: An integer specifying the new number of iterations between MCMC samples used for calculating posterior means.
	
Output:

	This macro does not produce any output datasets.
	
	The num_mcmc_iterations, num_burn, and num_thin variables in the _iterations dataset of multivar_mcmc_model are updated with the new values.
	
Details:

	Since multivar MCMC models save MCMC samples from every
	iteration, the number of iterations, burn-in, and thinning values used to
	calculate posterior means and covariances can be changed. This utility can
	be used to update those values while performing error-checking.

	For models with stored random effect (u) matrices from the main MCMC
	chain, not all u matrices are stored by default. As a result, the burn-in
	for models with main MCMC chain u matrices cannot be decreased, and the
	new thinning value must be a multiple of the original thinning value. An
	error will be thrown if u matrices are not found for the new thinned
	iterations.

	Since changing the number of iterations, burn-in, and thinning can alter
	the posterior means, post-MCMC 'u' matrix draws conditional on the
	posterior mean may not be valid and will throw a warning.
*/
%macro burn_and_thin(multivar_mcmc_lib=WORK,
										 multivar_mcmc_model=,
										 num_mcmc_iterations=,
										 num_burn=,
										 num_thin=);									
	
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._iters (rename=(num_mcmc_iterations=num_mcmc_iterations_old
																																 num_burn=num_burn_old
																																 num_thin=num_thin_old)
																												);
		
		%if &num_mcmc_iterations. = %str() %then %do;
			num_mcmc_iterations = num_mcmc_iterations_old;
		%end;
		%else %do;
			num_mcmc_iterations = &num_mcmc_iterations.;
		%end;
		
		%if &num_burn. = %str() %then %do;
			num_burn = num_burn_old;
		%end;
		%else %do;
			num_burn = &num_burn.;
		%end;
		
		%if &num_thin. = %str() %then %do;
			num_thin = num_thin_old;
		%end;
		%else %do;
			num_thin = &num_thin.;
		%end;
		
		if num_mcmc_iterations > num_trace then do;
		
			put "ERROR: Number of MCMC iterations cannot be higher than the number of traces in the model";
			abort abend 2;
		end;
		
		if num_burn >= num_mcmc_iterations then do;
		
			put "ERROR: Number of MCMC iterations must be greater than the number of burn-in iterations";
			abort abend 2;
		end;
		
		if num_post > 0 then do;
		
			put "WARNING: Post-MCMC U matrices may not be valid because posterior means may have changed";
		end;
		
		call symputx("num_mcmc_iterations", num_mcmc_iterations);
		call symputx("num_burn", num_burn);
		call symputx("num_thin", num_thin);
	run;
	
	%if %sysfunc(exist(&multivar_mcmc_lib..&multivar_mcmc_model._u_main)) = 1 %then %do;
	
		data _NULL_;
			set &multivar_mcmc_lib..&multivar_mcmc_model._u_main end=last_obs;
			by iteration;
			if iteration > &num_burn. and
				 iteration <= &num_mcmc_iterations. and
				 mod(iteration, &num_thin.) = mod(&num_burn. + 1, &num_thin.);
				 
			retain num_found;
			
			if _N_ = 1 then num_found = 0;
			
			if first.iteration then num_found = num_found + 1;
			
			if last_obs = 1 then do;
			
				num_expected = int((&num_mcmc_iterations. - (&num_burn. + 1))/&num_thin.) + 1;
				if num_found ^= num_expected then do;
				
					put "ERROR: MCMC chain U matrices not saved for requested iterations";
					abort abend 2;
				end;
			end;
		run;
	%end;
	
	**Set new total iterations, burn-in, and thinning;
	data &multivar_mcmc_lib..&multivar_mcmc_model._iters;
		set &multivar_mcmc_lib..&multivar_mcmc_model._iters;
		
		num_mcmc_iterations = &num_mcmc_iterations.;
		num_burn = &num_burn.;
		num_thin = &num_thin.;
	run;
%mend burn_and_thin;

/*
Draw Parameter Samples from an MCMC Chain

Description:

	This macro extracts samples of model parameters from an MCMC chain at a specified interval.
	
Parameters:

	- multivar_mcmc_lib: The SAS library that contains the output datasets of an MCMC model. (default = WORK)
	- multivar_mcmc_model: Prefix for the output datasets of the MCMC model (max 23 characters).
												 See nci_multivar_mcmc macro for more details.
	- num_draws:  integer specifying the number of parameter samples to draw from the MCMC chain. (default = 10)
	- draw_spacing: An integer specifying the number of iterations between samples. 
									If blank, defaults to 5 times the thinning value of the MCMC model. (default = blank)
	- outlib: The SAS library to store output datasets. (default = WORK)
	- outname: The prefix used to name output datasets (max 21 characters, default = DRAW).
	
Output:

	The following SAS data sets are created in outlib:
	
		outname1-outname&num_draws.: Multivar MCMC datasets for each parameter set that was drawn from multivar_mcmc_model.
																 For a complete listing of multivar MCMC datasets, see Output section for nci_multivar_mcmc.
	
Details:

	Drawing parameter values from an MCMC chain after the burn-in period
	is equivalent to drawing from the posterior distribution of the parameters. The
	variation in the MCMC chain is used to sample different parameter values as
	opposed to finding a covariance matrix. Using the MCMC chain has the
	advantage of using the actual posterior distribution of the parameters
	instead of making an assumption. Accurately sampling parameter values is
	required for measurement error correction methods such as multiple
	imputation.

	The function starts by drawing the parameter set from the first iteration
	after burn-in, and then draws parameter sets every draw_spacing
	iterations until num_draws sets have been drawn. If random effects (u)
	matrices from the main MCMC chain are saved, draw_spacing must be a
	multiple of the original thinning value of the MCMC model.
*/
%macro draw_parameters(multivar_mcmc_lib=WORK,
											 multivar_mcmc_model=,
											 num_draws=10,
											 draw_spacing=,
											 outlib=WORK,
											 outname=DRAW);
											 
	%local first_draw last_draw draw i
				 num_variables var_j
				 has_never_consumers
				 num_mcmc_iterations num_burn num_thin;
											 
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._iters;
		
		call symputx("num_mcmc_iterations", num_mcmc_iterations);
		call symputx("num_burn", num_burn);
		call symputx("num_thin", num_thin);
	run;
											 
	**default thinning for parameter draws to 5 times original thinning number;
	%if &draw_spacing. = %str() %then %do;
	
		data _NULL_;
		
			draw_spacing = 5*&num_thin.;
			call symputx("draw_spacing", draw_spacing);
		run;
	%end;
	
	**Check if enough iterations exist for parameter draws;
	data _NULL_;
	
		first_draw = &num_burn. + 1;
		last_draw = first_draw + &draw_spacing.*(&num_draws. - 1);
		
		if last > &num_mcmc_iterations. then do;
		
			put "ERROR: Not enough iterations after burn-in to make the requested number of parameter draws";
		end;
		
		call symputx("first_draw", first_draw);
		call symputx("last_draw", last_draw);
	run;
	
	%if %sysfunc(exist(&multivar_mcmc_lib..&multivar_mcmc_model._u_main)) = 1 %then %do;
	
		**Check if selected U matrices were saved;
		data _NULL_;
			set &multivar_mcmc_lib..&multivar_mcmc_model._u_main end=last_obs;
			by iteration;
			if iteration >= &first_draw. and
				 iteration <= &last_draw. and
				 mod(iteration, &draw_spacing.) = mod(&first_draw., &draw_spacing.);
				 
			retain num_found;
			
			if _N_ = 1 then num_found = 0;
			
			if first.iteration then num_found = num_found + 1;
			
			if last_obs = 1 then do;
			
				num_expected = &num_draws.;
				if num_found ^= num_expected then do;
				
					put "ERROR: MCMC chain U matrices not saved for requested draws";
					abort abend 2;
				end;
			end;
		run;
	%end;
	
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
		
		num_variables = 2*num_episodic + num_daily;
		call symputx("num_variables", num_variables);
		
		call symputx("has_never_consumers", has_never_consumers);
	run;
	
	%do i = 1 %to &num_draws.;
	
		data _NULL_;
		
			draw = &first_draw. + &draw_spacing.*(&i. - 1);
			call symputx("draw", draw);
		run;
		
		**draw parameters from specified iteration;
		%do var_j = 1 %to &num_variables.;
		
			data &outlib..&outname.&i._beta&var_j.;
				set &multivar_mcmc_lib..&multivar_mcmc_model._beta&var_j.;
				if _N_ = &draw.;
			run;
		%end;
		
		data &outlib..&outname.&i._sigma_e;
			set &multivar_mcmc_lib..&multivar_mcmc_model._sigma_e;
			if _N_ = &draw.;
		run;
		
		data &outlib..&outname.&i._sigma_u;
			set &multivar_mcmc_lib..&multivar_mcmc_model._sigma_u;
			if _N_ = &draw.;
		run;
		
		%if %sysfunc(exist(&multivar_mcmc_lib..&multivar_mcmc_model._u_main)) = 1 %then %do;
		
			data &outlib..&outname.&i._u_post (keep = post_mcmc_iteration u_col:);
				set &multivar_mcmc_lib..&multivar_mcmc_model._u_main;
				if iteration = &draw.;
				
				post_mcmc_iteration = 1;
			run;
		%end;
		
		%if &has_never_consumers. = 1 %then %do;
		
			data &outlib..&outname.&i._alpha1;
				set &multivar_mcmc_lib..&multivar_mcmc_model._alpha1;
				if _N_ = &draw.;
			run;
			
			data &outlib..&outname.&i._conprob1;
				set &multivar_mcmc_lib..&multivar_mcmc_model._conprob1;
				if _N_ = &draw.;
			run;
		%end;
		
		**create new NCI multivar MCMC object;
		data &outlib..&outname.&i._subjects;
			set &multivar_mcmc_lib..&multivar_mcmc_model._subjects;
		run;
		
		data &outlib..&outname.&i._vars;
			set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
		run;
		
		data &outlib..&outname.&i._covars;
			set &multivar_mcmc_lib..&multivar_mcmc_model._covars;
		run;
		
		%if &has_never_consumers. = 1 %then %do;
		
			data &outlib..&outname.&i._covarsnc;
				set &multivar_mcmc_lib..&multivar_mcmc_model._covarsnc;
			run;
		%end;
		
		data &outlib..&outname.&i._iters;
		
			num_trace = 1;
			
			num_mcmc_iterations = 1;
			num_burn = 0;
			num_thin = 1;
			%if %sysfunc(exist(&multivar_mcmc_lib..&multivar_mcmc_model._u_main)) = 1 %then %do;
				num_post = 1;
			%end;
			%else %do;
				num_post = 0;
			%end;
		run;
		
		data &outlib..&outname.&i._backtran;
			set &multivar_mcmc_lib..&multivar_mcmc_model._backtran;
		run;
	%end;
%mend draw_parameters;