%include "&sysincludefiledir./internal/mcmc_trace_plot_functions.sas";

/*
MCMC Trace Plots

Description:

	Generate trace plots from an NCI multivar MCMC model.
	
Parameter:

	- multivar_mcmc_lib: The SAS library that contains the output datasets of an MCMC model. (default = WORK)
	- multivar_mcmc_model: Prefix for the output datasets of the MCMC model (max 23 characters).
												 See nci_multivar_mcmc macro for more details.
	- do_beta: Flag to specify whether trace plots of fixed effect (beta) coefficients should be produced. (Y/N, default = Y)
	- do_sigma_e: Flag to specify whether trace plots of residual error covariance matrix (Sigma-e) elements should be produced. (Y/N, default = Y)
	- do_sigma_u: Flag to specify whether trace plots of random effect covariance matrix (Sigma-u) elements should be produced. (Y/N, default = Y)
	- do_sigma_overlay: Flag to specify whether overlays of Sigma-e and Sigma-u element trace plots should be produced. (Y/N, default = N)
	- do_alpha1: Flag to specify whether trace plots of never-consumer model (alpha1) coefficients should be produced. 
							 No effect if no never-consumer variable was modeled. (Y/N, default = Y)
	- do_consumer_probability: Flag to specify whether trace plots of mean consumer probability should be produced. 
														 No effect if no never-consumer variable was modeled. (Y/N, default = Y)
	- plot_type: Space-delimited list specifying what types of plots to generate.
							 One or both of the following can be specified:
							 	 - VALUE: Plots of a parameter's value at every MCMC iteration. This is the standard type of trace plot. (default)
							 	 - MEAN: Plots of the rolling mean of a parameter at thinned MCMC iterations after the burn-in period.
							 	 
Output:

	This macro does not produce any output datasets.
	
	The trace plots that are created are output using ODS.
	
Details:

	Trace plots are a method of assessing the convergence of MCMC chain.
	The value of an MCMC parameter (such as a coefficient or a covariance
	matrix element) is plotted against the number of iterations to see how the
	value fluctuates over time. Plots of the parameter values and rolling
	means can be produced. A parameter value that moves around a fixed point with
	a uniform variance over time is said to have converged or mixed well.
*/

%macro trace_plots(multivar_mcmc_lib=WORK,
									 multivar_mcmc_model=,
									 do_beta=Y,
									 do_sigma_e=Y,
									 do_sigma_u=Y,
									 do_sigma_overlay=N,
									 do_alpha1=Y,
									 do_consumer_probability=Y,
									 plot_type=VALUE);
									 
	%local variables num_episodic num_daily i
				 has_never_consumers
				 num_mcmc_iterations num_burn num_thin;
									 
	**types of plots to generate;
	%let plot_type = %upcase(&plot_type.);
	
	**number of iterations and burn-in;
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._iters;
		
		call symputx("num_mcmc_iterations", num_mcmc_iterations);
		call symputx("num_burn", num_burn);
		call symputx("num_thin", num_thin);
	run;
	
	**get variables to make trace plots for;
	data _NULL_;
			set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
		
			call symputx("num_episodic", num_episodic);
			call symputx("num_daily", num_daily);
		run;
		
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
			
		length variables $32767.;
			
		variables = "";
			
		%do i = 1 %to &num_episodic.;
			variables = catx(" ", variables, episodic_ind&i.);
			variables = catx(" ", variables, episodic_amt&i.);
		%end;
			
		%do i = 1 %to &num_daily.;
			variables = catx(" ", variables, daily_amt&i.);
		%end;
			
		call symputx("variables", variables);
	run;
	
	**beta trace plots;
	%if %upcase(&do_beta.) = Y %then %do;
	
		%beta_trace_plots(beta=&multivar_mcmc_lib..&multivar_mcmc_model._beta,
											variables=&variables.,
											covariate_data=&multivar_mcmc_lib..&multivar_mcmc_model._covars,
											num_mcmc_iterations=&num_mcmc_iterations.,
											num_burn=&num_burn.,
											num_thin=&num_thin.,
											plot_type=&plot_type.);
	%end;
	
	**Sigma-e trace plots;
	%if %upcase(&do_sigma_e.) = Y %then %do;
	
		%sigma_trace_plots(sigma=&multivar_mcmc_lib..&multivar_mcmc_model._sigma_e,
											 component=e,
											 variables=&variables.,
											 num_mcmc_iterations=&num_mcmc_iterations.,
											 num_burn=&num_burn.,
											 num_thin=&num_thin.,
											 plot_type=&plot_type.);
	%end;
	
	**Sigma-u trace plots;
	%if %upcase(&do_sigma_u.) = Y %then %do;
	
		%sigma_trace_plots(sigma=&multivar_mcmc_lib..&multivar_mcmc_model._sigma_u,
											 component=u,
											 variables=&variables.,
											 num_mcmc_iterations=&num_mcmc_iterations.,
											 num_burn=&num_burn.,
											 num_thin=&num_thin.,
											 plot_type=&plot_type.);
	%end;
	
	**sigma overlay plots;
	%if %upcase(&do_sigma_overlay.) = Y %then %do;
	
		%sigma_overlay_plots(sigma_e=&multivar_mcmc_lib..&multivar_mcmc_model._sigma_e,
											 	 sigma_u=&multivar_mcmc_lib..&multivar_mcmc_model._sigma_u,
											 	 variables=&variables.,
											 	 num_mcmc_iterations=&num_mcmc_iterations.,
											 	 num_burn=&num_burn.,
											 	 num_thin=&num_thin.,
											 	 plot_type=&plot_type.);
	%end;
	
	**Never-consumer trace plots;
	data _NULL_;
		set &multivar_mcmc_lib..&multivar_mcmc_model._vars;
		
		call symputx("has_never_consumers", has_never_consumers);
	run;
	
	%if &has_never_consumers. = 1 %then %do;
	
		**alpha1 trace plots;
		%if %upcase(&do_alpha1.) = Y %then %do;
		
			%alpha1_trace_plots(alpha1=&multivar_mcmc_lib..&multivar_mcmc_model._alpha1,
													never_consumer_covariate_data=&multivar_mcmc_lib..&multivar_mcmc_model._covarsnc,
													num_mcmc_iterations=&num_mcmc_iterations.,
													num_burn=&num_burn.,
													num_thin=&num_thin.,
													plot_type=&plot_type.);
		%end;
		
		**consumer probability trace plots;
		%if %upcase(&do_consumer_probability.) = Y %then %do;
		
			%consumer_probability_trace_plots(consumer_probabilities=&multivar_mcmc_lib..&multivar_mcmc_model._conprob1,
																				num_mcmc_iterations=&num_mcmc_iterations.,
																				num_burn=&num_burn.,
																				num_thin=&num_thin.,
																				plot_type=&plot_type.);
		%end;
	%end;
%mend trace_plots;