#Scoring for adequacy components (higher intake is better)
score_adequacy <- function(intake,
                           min.intake,
                           max.intake,
                           max.score) {

  score <- ifelse(intake >= min.intake,
                  ifelse(intake <= max.intake,
                         max.score*(intake - min.intake)/(max.intake - min.intake),
                         max.score),
                  0)
}

#Scoring for moderation components (lower intake is better)
score_moderation <- function(intake,
                             min.intake,
                             max.intake,
                             max.score) {

  score <- ifelse(intake >= min.intake,
                  ifelse(intake <= max.intake,
                         max.score - max.score*(intake - min.intake)/(max.intake - min.intake),
                         0),
                  max.score)
}

# HEI-2020 Scoring
#
# Description:
# Calculates total and component HEI-2020 scores from usual intakes of foods and
# nutrients. It is intended for populations age 2 and older.
#
# Details:
# The HEI-2020 score is intended for populations age 2 and older. For
# populations age 12 - 23 months, the HEI-2020 Toddler score is recommended.
#
# This function is intended for use on usual intakes of foods and nutrients
# after measurement error correction. It does not perform any measurement
# error correction its own, and it is not intended for use directly on
# uncorrected data (such as raw NHANES data).
#
#
# Input
# * input.data: A data frame
# * kcal: Energy (kcal) variable
# * v.totalleg: Total vegetables (cup eq.) variable
# * v.drkgrleg: Dark green vegetables and legumes (cup eq.) variable
# * f.total: Total fruit (cup eq.) variable
# * f.wholefrt: Whole fruit (cup eq.) variable
# * g.whole: Whole grain (oz. eq.) variable
# * d.total: Total dairy (cup eq.) variable
# * pf.allprotleg: Total protein foods (cup eq.) variable
# * pf.seaplantleg: Seafood and plant protein and legumes (cup eq.) variable
# * monopoly: Monounsaturated and polyunsaturated fat (g) variable
# * satfat: Saturated fat (g) variable
# * sodium: Sodium (mg) variable
# * g.refined: Refined grain (oz. eq.) variable
# * add.sugars: Added sugars (tsp. eq.) variable
# * id: Unique subject identifier variable, optional
# * weight: Weight variable, optional
# * additional.output: Vector of additional variables to pass through to output
#
#
# Output: A data frame with the following columns:
# * id: Unique subject identifier, if provided
# * weight: Weight, if provided
# * additional.output: Additional variables, if provided
# * HEI2020_TOTAL_SCORE: Total HEI-2020 score
# * HEI2020_TOTAL_FRUIT: HEI-2020 total fruit component
# * HEI2020_WHOLE_FRUIT: HEI-2020 whole fruit component
# * HEI2020_TOTAL_VEGETABLES: HEI-2020 total vegetables component
# * HEI2020_GREENS_AND_BEANS: HEI-2020 greens and beans component
# * HEI2020_WHOLE_GRAIN: HEI-2020 whole grain component
# * HEI2020_TOTAL_DAIRY: HEI-2020 total dairy component
# * HEI2020_TOTAL_PROTEIN: HEI-2020 total protein component
# * HEI2020_SEAFOOD_PLANT_PROTEIN: HEI-2020 seafood and plant protein component
# * HEI2020_FATTY_ACID: HEI-2020 fatty acid component
# * HEI2020_REFINED_GRAIN: HEI-2020 refined grain component
# * HEI2020_SODIUM: HEI-2020 sodium component
# * HEI2020_ADDED_SUGARS: HEI-2020 added sugars component
# * HEI2020_SATURATED_FAT: HEI-2020 saturated fat component
hei2020 <- function(input.data,
                    kcal,
                    v.totalleg,
                    v.drkgrleg,
                    f.total,
                    f.wholefrt,
                    g.whole,
                    d.total,
                    pf.allprotleg,
                    pf.seaplantleg,
                    monopoly,
                    satfat,
                    sodium,
                    g.refined,
                    add.sugars,
                    id=NULL,
                    weight=NULL,
                    additional.output=NULL) {

  #Total Fruit
  total.fruit.min <- 0
  total.fruit.max <- 0.8

  total.fruit.density <- input.data[,f.total]/(input.data[,kcal]/1000)

  hei2020.total.fruit <- score_adequacy(total.fruit.density, total.fruit.min, total.fruit.max, 5)

  #Whole Fruit
  whole.fruit.min <- 0
  whole.fruit.max <- 0.4

  whole.fruit.density <- input.data[,f.wholefrt]/(input.data[,kcal]/1000)

  hei2020.whole.fruit <- score_adequacy(whole.fruit.density, whole.fruit.min, whole.fruit.max, 5)

  #Total Vegetables
  total.vegetables.min <- 0
  total.vegetables.max <- 1.1

  total.vegetables.density <- input.data[,v.totalleg]/(input.data[,kcal]/1000)

  hei2020.total.vegetables <- score_adequacy(total.vegetables.density, total.vegetables.min, total.vegetables.max, 5)

  #Greens and Beans
  greens.and.beans.min <- 0
  greens.and.beans.max <- 0.2

  greens.and.beans.density <- input.data[,v.drkgrleg]/(input.data[,kcal]/1000)

  hei2020.greens.and.beans <- score_adequacy(greens.and.beans.density, greens.and.beans.min, greens.and.beans.max, 5)

  #Whole Grains
  whole.grain.min <- 0
  whole.grain.max <- 1.5

  whole.grain.density <- input.data[,g.whole]/(input.data[,kcal]/1000)

  hei2020.whole.grain <- score_adequacy(whole.grain.density, whole.grain.min, whole.grain.max, 10)

  #Total Dairy
  total.dairy.min <- 0
  total.dairy.max <- 1.3

  total.dairy.density <- input.data[,d.total]/(input.data[,kcal]/1000)

  hei2020.total.dairy <- score_adequacy(total.dairy.density, total.dairy.min, total.dairy.max, 10)

  #Total Protein
  total.protein.min <- 0
  total.protein.max <- 2.5

  total.protein.density <- input.data[,pf.allprotleg]/(input.data[,kcal]/1000)

  hei2020.total.protein <- score_adequacy(total.protein.density, total.protein.min, total.protein.max, 5)

  #Seafood and Plant Protein
  seafood.plant.protein.min <- 0
  seafood.plant.protein.max <- 0.8

  seafood.plant.protein.density <- input.data[,pf.seaplantleg]/(input.data[,kcal]/1000)

  hei2020.seafood.plant.protein <- score_adequacy(seafood.plant.protein.density, seafood.plant.protein.min, seafood.plant.protein.max, 5)

  #Fatty Acids
  fatty.acid.min <- 1.2
  fatty.acid.max <- 2.5

  fatty.acid.ratio <- input.data[,monopoly]/input.data[,satfat]

  hei2020.fatty.acid <- score_adequacy(fatty.acid.ratio, fatty.acid.min, fatty.acid.max, 10)
  hei2020.fatty.acid[input.data[,satfat] == 0 & input.data[,monopoly] == 0] <- 0
  hei2020.fatty.acid[input.data[,satfat] == 0 & input.data[,monopoly] > 0] <- 10

  #Refined Grain
  refined.grain.min <- 1.8
  refined.grain.max <- 4.3

  refined.grain.density <- input.data[,g.refined]/(input.data[,kcal]/1000)

  hei2020.refined.grain <- score_moderation(refined.grain.density, refined.grain.min, refined.grain.max, 10)

  #Sodium
  sodium.density.min <- 1.1
  sodium.density.max <- 2.0

  sodium.density <- input.data[,sodium]/input.data[,kcal]

  hei2020.sodium <- score_moderation(sodium.density, sodium.density.min, sodium.density.max, 10)

  #Added Sugars
  added.sugars.min <- 6.5
  added.sugars.max <- 26

  added.sugars.percent <- 100*(input.data[,add.sugars]*16/input.data[,kcal])

  hei2020.added.sugars <- score_moderation(added.sugars.percent, added.sugars.min, added.sugars.max, 10)

  #Saturated Fat
  saturated.fat.min <- 8
  saturated.fat.max <- 16

  saturated.fat.percent <- 100*(input.data[,satfat]*9/input.data[,kcal])

  hei2020.saturated.fat <- score_moderation(saturated.fat.percent, saturated.fat.min, saturated.fat.max, 10)

  #Set HEI scores to 0 when no energy consumption is recorded
  hei2020.total.fruit[input.data[,kcal] == 0] <- 0
  hei2020.whole.fruit[input.data[,kcal] == 0] <- 0
  hei2020.total.vegetables[input.data[,kcal] == 0] <- 0
  hei2020.greens.and.beans[input.data[,kcal] == 0] <- 0
  hei2020.whole.grain[input.data[,kcal] == 0] <- 0
  hei2020.total.dairy[input.data[,kcal] == 0] <- 0
  hei2020.total.protein[input.data[,kcal] == 0] <- 0
  hei2020.seafood.plant.protein[input.data[,kcal] == 0] <- 0
  hei2020.fatty.acid[input.data[,kcal] == 0] <- 0
  hei2020.refined.grain[input.data[,kcal] == 0] <- 0
  hei2020.sodium[input.data[,kcal] == 0] <- 0
  hei2020.added.sugars[input.data[,kcal] == 0] <- 0
  hei2020.saturated.fat[input.data[,kcal] == 0] <- 0

  #Calculate total HEI score by adding components together
  hei2020.total.score <- hei2020.total.fruit + hei2020.whole.fruit + hei2020.total.vegetables + hei2020.greens.and.beans +
                         hei2020.whole.grain + hei2020.total.dairy + hei2020.total.protein + hei2020.seafood.plant.protein +
                         hei2020.fatty.acid + hei2020.refined.grain + hei2020.sodium + hei2020.added.sugars + hei2020.saturated.fat

  #output scores
  hei2020.scores <- data.frame(input.data[,id,drop=FALSE],
                               input.data[,weight,drop=FALSE],
                               input.data[,additional.output,drop=FALSE],
                               HEI2020_TOTAL_SCORE=hei2020.total.score,
                               HEI2020_TOTAL_FRUIT=hei2020.total.fruit,
                               HEI2020_WHOLE_FRUIT=hei2020.whole.fruit,
                               HEI2020_TOTAL_VEGETABLES=hei2020.total.vegetables,
                               HEI2020_GREENS_AND_BEANS=hei2020.greens.and.beans,
                               HEI2020_WHOLE_GRAIN=hei2020.whole.grain,
                               HEI2020_TOTAL_DAIRY=hei2020.total.dairy,
                               HEI2020_TOTAL_PROTEIN=hei2020.total.protein,
                               HEI2020_SEAFOOD_PLANT_PROTEIN=hei2020.seafood.plant.protein,
                               HEI2020_FATTY_ACID=hei2020.fatty.acid,
                               HEI2020_REFINED_GRAIN=hei2020.refined.grain,
                               HEI2020_SODIUM=hei2020.sodium,
                               HEI2020_ADDED_SUGARS=hei2020.added.sugars,
                               HEI2020_SATURATED_FAT=hei2020.saturated.fat)

  return(hei2020.scores)
}

# HEI-2020 Toddler Scoring
#
# Description:
# Calculates total and component HEI-2020 Toddler scores from usual intakes of
# foods and nutrients. It is intended for populations age 12 - 23 months.
#
# Details:
# The HEI-2020 Toddler score is intended for populations age 12 - 23 months.
# For populations age 2 and older, the HEI-2020 score is recommended.
#
# This function is intended for use on usual intakes of foods and nutrients
# after measurement error correction. It does not perform any measurement
# error correction its own, and it is not intended for use directly on
# uncorrected data (such as raw NHANES data).
#
#
# Input
# * input.data: A data frame
# * kcal: Energy (kcal) variable
# * v.totalleg: Total vegetables (cup eq.) variable
# * v.drkgrleg: Dark green vegetables and legumes (cup eq.) variable
# * f.total: Total fruit (cup eq.) variable
# * f.wholefrt: Whole fruit (cup eq.) variable
# * g.whole: Whole grain (oz. eq.) variable
# * d.total: Total dairy (cup eq.) variable
# * pf.allprotleg: Total protein foods (cup eq.) variable
# * pf.seaplantleg: Seafood and plant protein and legumes (cup eq.) variable
# * monopoly: Monounsaturated and polyunsaturated fat (g) variable
# * satfat: Saturated fat (g) variable
# * sodium: Sodium (mg) variable
# * g.refined: Refined grain (oz. eq.) variable
# * add.sugars: Added sugars (tsp. eq.) variable
# * id: Unique subject identifier variable, optional
# * weight: Weight variable, optional
# * additional.output: Vector of additional variables to pass through to output
#
#
# Output: A data frame with the following columns:
# * id: Unique subject identifier, if provided
# * weight: Weight, if provided
# * additional.output: Additional variables, if provided
# * HEITODL20_TOTAL_SCORE: Total HEI-2020 Toddler score
# * HEITODL20_TOTAL_FRUIT: HEI-2020 Toddler total fruit component
# * HEITODL20_WHOLE_FRUIT: HEI-2020 Toddler whole fruit component
# * HEITODL20_TOTAL_VEGETABLES: HEI-2020 Toddler total vegetables component
# * HEITODL20_GREENS_AND_BEANS: HEI-2020 Toddler greens and beans component
# * HEITODL20_WHOLE_GRAIN: HEI-2020 Toddler whole grain component
# * HEITODL20_TOTAL_DAIRY: HEI-2020 Toddler total dairy component
# * HEITODL20_TOTAL_PROTEIN: HEI-2020 Toddler total protein component
# * HEITODL20_SEAFOOD_PLANT_PROTEIN: HEI-2020 Toddler seafood and plant protein component
# * HEITODL20_FATTY_ACID: HEI-2020 Toddler fatty acid component
# * HEITODL20_REFINED_GRAIN: HEI-2020 Toddler refined grain component
# * HEITODL20_SODIUM: HEI-2020 Toddler sodium component
# * HEITODL20_ADDED_SUGARS: HEI-2020 Toddler added sugars component
# * HEITODL20_SATURATED_FAT: HEI-2020 Toddler saturated fat component
hei2020_toddler <- function(input.data,
                            kcal,
                            v.totalleg,
                            v.drkgrleg,
                            f.total,
                            f.wholefrt,
                            g.whole,
                            d.total,
                            pf.allprotleg,
                            pf.seaplantleg,
                            monopoly,
                            satfat,
                            sodium,
                            g.refined,
                            add.sugars,
                            id=NULL,
                            weight=NULL,
                            additional.output=NULL) {

  #Total Fruit
  total.fruit.min <- 0
  total.fruit.max <- 0.7

  total.fruit.density <- input.data[,f.total]/(input.data[,kcal]/1000)

  heitodl20.total.fruit <- score_adequacy(total.fruit.density, total.fruit.min, total.fruit.max, 5)

  #Whole Fruit
  whole.fruit.min <- 0
  whole.fruit.max <- 0.3

  whole.fruit.density <- input.data[,f.wholefrt]/(input.data[,kcal]/1000)

  heitodl20.whole.fruit <- score_adequacy(whole.fruit.density, whole.fruit.min, whole.fruit.max, 5)

  #Total Vegetables
  total.vegetables.min <- 0
  total.vegetables.max <- 0.9

  total.vegetables.density <- input.data[,v.totalleg]/(input.data[,kcal]/1000)

  heitodl20.total.vegetables <- score_adequacy(total.vegetables.density, total.vegetables.min, total.vegetables.max, 5)

  #Greens and Beans
  greens.and.beans.min <- 0
  greens.and.beans.max <- 0.1

  greens.and.beans.density <- input.data[,v.drkgrleg]/(input.data[,kcal]/1000)

  heitodl20.greens.and.beans <- score_adequacy(greens.and.beans.density, greens.and.beans.min, greens.and.beans.max, 5)

  #Whole Grains
  whole.grain.min <- 0
  whole.grain.max <- 1.5

  whole.grain.density <- input.data[,g.whole]/(input.data[,kcal]/1000)

  heitodl20.whole.grain <- score_adequacy(whole.grain.density, whole.grain.min, whole.grain.max, 10)

  #Total Dairy
  total.dairy.min <- 0
  total.dairy.max <- 2

  total.dairy.density <- input.data[,d.total]/(input.data[,kcal]/1000)

  heitodl20.total.dairy <- score_adequacy(total.dairy.density, total.dairy.min, total.dairy.max, 10)

  #Total Protein
  total.protein.min <- 0
  total.protein.max <- 2.0

  total.protein.density <- input.data[,pf.allprotleg]/(input.data[,kcal]/1000)

  heitodl20.total.protein <- score_adequacy(total.protein.density, total.protein.min, total.protein.max, 5)

  #Seafood and Plant Protein
  seafood.plant.protein.min <- 0
  seafood.plant.protein.max <- 0.5

  seafood.plant.protein.density <- input.data[,pf.seaplantleg]/(input.data[,kcal]/1000)

  heitodl20.seafood.plant.protein <- score_adequacy(seafood.plant.protein.density, seafood.plant.protein.min, seafood.plant.protein.max, 5)

  #Fatty Acids
  fatty.acid.min <- 0.9
  fatty.acid.max <- 1.5

  fatty.acid.ratio <- input.data[,monopoly]/input.data[,satfat]

  heitodl20.fatty.acid <- score_adequacy(fatty.acid.ratio, fatty.acid.min, fatty.acid.max, 10)
  heitodl20.fatty.acid[input.data[,satfat] == 0 & input.data[,monopoly] == 0] <- 0
  heitodl20.fatty.acid[input.data[,satfat] == 0 & input.data[,monopoly] > 0] <- 10

  #Refined Grain
  refined.grain.min <- 1.5
  refined.grain.max <- 3.4

  refined.grain.density <- input.data[,g.refined]/(input.data[,kcal]/1000)

  heitodl20.refined.grain <- score_moderation(refined.grain.density, refined.grain.min, refined.grain.max, 10)

  #Sodium
  sodium.density.min <- 1.1
  sodium.density.max <- 1.7

  sodium.density <- input.data[,sodium]/input.data[,kcal]

  heitodl20.sodium <- score_moderation(sodium.density, sodium.density.min, sodium.density.max, 10)

  #Added Sugars
  added.sugars.min <- 0
  added.sugars.max <- 13.8

  added.sugars.percent <- 100*(input.data[,add.sugars]*16/input.data[,kcal])

  heitodl20.added.sugars <- score_moderation(added.sugars.percent, added.sugars.min, added.sugars.max, 10)

  #Saturated Fat
  saturated.fat.min <- 12.2
  saturated.fat.max <- 18.2

  saturated.fat.percent <- 100*(input.data[,satfat]*9/input.data[,kcal])

  heitodl20.saturated.fat <- score_moderation(saturated.fat.percent, saturated.fat.min, saturated.fat.max, 10)

  #Set HEI scores to 0 when no energy consumption is recorded
  heitodl20.total.fruit[input.data[,kcal] == 0] <- 0
  heitodl20.whole.fruit[input.data[,kcal] == 0] <- 0
  heitodl20.total.vegetables[input.data[,kcal] == 0] <- 0
  heitodl20.greens.and.beans[input.data[,kcal] == 0] <- 0
  heitodl20.whole.grain[input.data[,kcal] == 0] <- 0
  heitodl20.total.dairy[input.data[,kcal] == 0] <- 0
  heitodl20.total.protein[input.data[,kcal] == 0] <- 0
  heitodl20.seafood.plant.protein[input.data[,kcal] == 0] <- 0
  heitodl20.fatty.acid[input.data[,kcal] == 0] <- 0
  heitodl20.refined.grain[input.data[,kcal] == 0] <- 0
  heitodl20.sodium[input.data[,kcal] == 0] <- 0
  heitodl20.added.sugars[input.data[,kcal] == 0] <- 0
  heitodl20.saturated.fat[input.data[,kcal] == 0] <- 0

  #Calculate total HEI score by adding components together
  heitodl20.total.score <- heitodl20.total.fruit + heitodl20.whole.fruit + heitodl20.total.vegetables + heitodl20.greens.and.beans +
                           heitodl20.whole.grain + heitodl20.total.dairy + heitodl20.total.protein + heitodl20.seafood.plant.protein +
                           heitodl20.fatty.acid + heitodl20.refined.grain + heitodl20.sodium + heitodl20.added.sugars + heitodl20.saturated.fat

  #output scores
  heitodl20.scores <- data.frame(input.data[,id,drop=FALSE],
                                 input.data[,weight,drop=FALSE],
                                 input.data[,additional.output,drop=FALSE],
                                 HEITODL20_TOTAL_SCORE=heitodl20.total.score,
                                 HEITODL20_TOTAL_FRUIT=heitodl20.total.fruit,
                                 HEITODL20_WHOLE_FRUIT=heitodl20.whole.fruit,
                                 HEITODL20_TOTAL_VEGETABLES=heitodl20.total.vegetables,
                                 HEITODL20_GREENS_AND_BEANS=heitodl20.greens.and.beans,
                                 HEITODL20_WHOLE_GRAIN=heitodl20.whole.grain,
                                 HEITODL20_TOTAL_DAIRY=heitodl20.total.dairy,
                                 HEITODL20_TOTAL_PROTEIN=heitodl20.total.protein,
                                 HEITODL20_SEAFOOD_PLANT_PROTEIN=heitodl20.seafood.plant.protein,
                                 HEITODL20_FATTY_ACID=heitodl20.fatty.acid,
                                 HEITODL20_REFINED_GRAIN=heitodl20.refined.grain,
                                 HEITODL20_SODIUM=heitodl20.sodium,
                                 HEITODL20_ADDED_SUGARS=heitodl20.added.sugars,
                                 HEITODL20_SATURATED_FAT=heitodl20.saturated.fat)

  return(heitodl20.scores)
}
